/*
 * Copyright 2017-2022 Lenses.io Ltd
 */
package io.lenses.sql.udf.value;

import io.lenses.sql.udf.UdfException;
import io.lenses.sql.udf.datatype.DataType;

import java.math.BigDecimal;
import java.math.BigInteger;

/**
 * The time-micros logical type represents a time of day, with no reference to a particular calendar, time zone or date, with a precision of one microsecond.
 *
 * A time-micros stores the number of microseconds after midnight, 00:00:00.000000.
 */
public class TimeMicrosValue extends Primitive<Long> {

    public TimeMicrosValue(long value) {
        super(value, DataType.ltTimeMicros());
    }

    @Override
    public BigDecimalValue toBigDecimalValue() throws UdfException {
        TryConversion<BigDecimalValue> c = () -> new BigDecimalValue(new BigDecimal(underlying));
        return convert(c, DataType.ltBigDecimal());
    }

    @Override
    public BigIntValue toBigIntValue() throws UdfException {
        TryConversion<BigIntValue> c = () -> new BigIntValue(BigInteger.valueOf(underlying));
        return convert(c, DataType.ltBigInt());
    }

    @Override
    public BooleanValue toBooleanValue() throws UdfException {
        throw UdfException.conversionNotSupported(dataType, DataType.ltBoolean());
    }

    @Override
    public ByteValue toByteValue() throws UdfException {
        TryConversion<ByteValue> c = () -> new ByteValue(underlying.byteValue());
        return convert(c, DataType.ltByte());
    }

    @Override
    public DoubleValue toDoubleValue() throws UdfException {
        TryConversion<DoubleValue> c = () -> new DoubleValue(underlying.doubleValue());
        return convert(c, DataType.ltDouble());
    }

    @Override
    public FloatValue toFloatValue() throws UdfException {
        TryConversion<FloatValue> c = () -> new FloatValue(underlying.floatValue());
        return convert(c, DataType.ltFloat());
    }

    @Override
    public IntValue toIntValue() throws UdfException {
        TryConversion<IntValue> c = () -> new IntValue(underlying.intValue());
        return convert(c, DataType.ltInt());
    }

    @Override
    public LongValue toLongValue() {
        return new LongValue(underlying);
    }

    @Override
    public StringValue toStringValue() throws UdfException {
        TryConversion<StringValue> c = () -> new StringValue(underlying.toString());
        return convert(c, DataType.ltString());
    }

}
