/*
 * Copyright 2017-2022 Lenses.io Ltd
 */
package io.lenses.sql.udf.value;

import io.lenses.sql.udf.UdfException;
import io.lenses.sql.udf.datatype.DataType;

import java.math.BigDecimal;
import java.math.BigInteger;

public class StringValue extends Primitive<String> {

    public StringValue(String value) {
        super(value, DataType.ltString());
        this.underlying = value;
    }

    @Override
    public BigDecimalValue toBigDecimalValue() throws UdfException {
        TryConversion<BigDecimalValue> c = () -> new BigDecimalValue(new BigDecimal(underlying));
        return convert(c, DataType.ltBigDecimal());
    }

    @Override
    public BigIntValue toBigIntValue() throws UdfException {
        TryConversion<BigIntValue> c = () -> new BigIntValue(BigInteger.valueOf(Long.parseLong(underlying)));
        return convert(c, DataType.ltBigInt());
    }

    @Override
    public BooleanValue toBooleanValue() throws UdfException {
        TryConversion<BooleanValue> c = () -> new BooleanValue(Boolean.parseBoolean(underlying));
        return convert(c, DataType.ltBoolean());
    }

    @Override
    public ByteValue toByteValue() throws UdfException {
        throw UdfException.conversionNotSupported(dataType, DataType.ltByte());
    }

    @Override
    public DoubleValue toDoubleValue() throws UdfException {
        TryConversion<DoubleValue> c = () -> new DoubleValue(Double.parseDouble(underlying));
        return convert(c, DataType.ltDouble());
    }

    @Override
    public FloatValue toFloatValue() throws UdfException {
        TryConversion<FloatValue> c = () -> new FloatValue(Float.parseFloat(underlying));
        return convert(c, DataType.ltFloat());
    }

    @Override
    public IntValue toIntValue() throws UdfException {
        TryConversion<IntValue> c = () -> new IntValue(Integer.parseInt(underlying));
        return convert(c, DataType.ltInt());
    }

    @Override
    public LongValue toLongValue() throws UdfException {
        TryConversion<LongValue> c = () -> new LongValue(Long.parseLong(underlying));
        return convert(c, DataType.ltLong());
    }

    @Override
    public StringValue toStringValue() {
        return this;
    }

}
