/*
 * Copyright 2017-2022 Lenses.io Ltd
 */
package io.lenses.sql.udf.value;

import io.lenses.sql.udf.UdfException;
import io.lenses.sql.udf.UdfRuntimeException;
import io.lenses.sql.udf.datatype.DataType;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Vector;

public class RepeatedValue<A extends Value> extends Container {

    private final List<A> underlying;
    private final DataType valueType;

    public RepeatedValue(List<A> values, DataType valueType) {
        super(DataType.ltRepeated(valueType));
        Optional<UdfRuntimeException> validationError = validate(values, valueType);
        if (validationError.isPresent()) {
            throw validationError.get();
        }
        this.underlying = values;
        this.valueType = valueType;
    }

    @Override
    public List<A> get() {
        return underlying;
    }

    @Override
    public RepeatedValue asRepeatedValue() {
        return this;
    }

    @Override
    public StructValue asStructValue() throws UdfException {
        throw new UdfException("Value " + this + " is not a struct.");
    }

    public Collection<A> getAllValues() {
        return underlying;
    }

    public Value get(int pos) {
        return underlying.get(pos);
    }

    public DataType getValueType() {
        return valueType;
    }

    public static <T extends Value> RepeatedValue<T> empty(DataType valueType) {
        List<T> empty = new Vector<T>();
        return new RepeatedValue<>(empty, valueType);
    }

    public static <T extends Value> RepeatedValue<T> ofOne(T first) {
        Vector<T> values = new Vector<>();
        values.add(first);
        return new RepeatedValue<>(values, first.dataType);
    }

    public static <T extends Value> RepeatedValue<T> ofTwo(T first, T second) {
        Vector<T> values = new Vector<>();
        values.add(first);
        values.add(second);
        if (!ofSameType(values)) throw UdfRuntimeException.invalidValueTypes();
        return new RepeatedValue<>(values, first.dataType);
    }

    public static <T extends Value> RepeatedValue<T> ofThree(T first, T second, T third) {
        Vector<T> values = new Vector<>();
        values.add(first);
        values.add(second);
        values.add(third);
        if (!ofSameType(values)) throw UdfRuntimeException.invalidValueTypes();
        return new RepeatedValue<>(values, first.dataType);
    }

    public static <T extends Value> RepeatedValue<T> ofValues(List<T> values) {
        DataType valueType;
        if (values.isEmpty()) {
            throw new UdfRuntimeException("List of values must not be empty.");
        } else if (!ofSameType(values)) {
            throw UdfRuntimeException.invalidValueTypes();
        } else {
            valueType = values.get(0).getDataType();
        }
        return new RepeatedValue<>(values, valueType);
    }

    private Optional<UdfRuntimeException> validate(List<A> values, DataType valueType) {
        Optional<UdfRuntimeException> result = Optional.empty();
        if (!values.isEmpty()) {
            if (!ofSameType(values)) {
                result = Optional.of(UdfRuntimeException.invalidValueTypes());
            }
            DataType firstDataType = values.get(0).getDataType();
            if (!firstDataType.equals(valueType)) {
                result = Optional.of(UdfRuntimeException.valueTypesMismatch(firstDataType, valueType));
            }
        }
        return result;
    }

    // TODO not very elegant => investigate ways to describe this with types while not complicating the API for users
    private static <T extends Value> boolean ofSameType(List<T> values) {
        boolean result = true;
        if (!values.isEmpty()) {
            String valueType = values.get(0).getDataType().name;
            for (Value v : values) {
                if (v.getDataType().name != valueType) result = false;
            }
        }
        return result;
    }

}
