/*
 * Copyright 2017-2022 Lenses.io Ltd
 */
package io.lenses.sql.udf.value;

import io.lenses.sql.udf.UdfException;
import io.lenses.sql.udf.datatype.DataType;

import java.util.Optional;

public class OptionalValue extends Value {
    private final Optional<Value> inner;

    protected OptionalValue(Optional<Value> inner, DataType dataType) throws UdfException {
        super(dataType);
        this.inner = inner;
    }

    public Boolean nonEmpty() {
        return inner.isPresent();
    }

    public Boolean isEmpty() {
        return !nonEmpty();
    }

    @Override
    public Object get() {
        return inner.orElse(null);
    }

    @Override
    public Primitive asPrimitive() throws UdfException {
        throwIfEmptyOnConversion("Primitive");
        return inner.get().asPrimitive();
    }

    @Override
    public Container asContainer() throws UdfException {
        throwIfEmptyOnConversion("Container");
        return inner.get().asContainer();
    }

    @Override
    public RepeatedValue asRepeatedValue() throws UdfException {
        throwIfEmptyOnConversion("RepeatedValue");
        return inner.get().asRepeatedValue();
    }

    @Override
    public StructValue asStructValue() throws UdfException {
        throwIfEmptyOnConversion("StructValue");
        return inner.get().asStructValue();
    }

    @Override
    public boolean isPrimitive() {
        return inner.isPresent() && inner.get().isPrimitive();
    }

    @Override
    public BigDecimalValue toBigDecimalValue() throws UdfException {
        throwIfEmptyOnConversion("BigDecimalValue");
        return inner.get().toBigDecimalValue();
    }

    @Override
    public BigIntValue toBigIntValue() throws UdfException {
        throwIfEmptyOnConversion("BigIntValue");
        return inner.get().toBigIntValue();
    }

    @Override
    public BooleanValue toBooleanValue() throws UdfException {
        throwIfEmptyOnConversion("BooleanValue");
        return inner.get().toBooleanValue();
    }

    @Override
    public ByteValue toByteValue() throws UdfException {
        throwIfEmptyOnConversion("ByteValue");
        return inner.get().toByteValue();
    }

    @Override
    public DoubleValue toDoubleValue() throws UdfException {
        throwIfEmptyOnConversion("DoubleValue");
        return inner.get().toDoubleValue();
    }

    @Override
    public FloatValue toFloatValue() throws UdfException {
        throwIfEmptyOnConversion("FloatValue");
        return inner.get().toFloatValue();
    }

    @Override
    public IntValue toIntValue() throws UdfException {
        throwIfEmptyOnConversion("IntValue");
        return inner.get().toIntValue();
    }

    @Override
    public LongValue toLongValue() throws UdfException {
        throwIfEmptyOnConversion("LongValue");
        return inner.get().toLongValue();
    }

    @Override
    public StringValue toStringValue() throws UdfException {
        throwIfEmptyOnConversion("StringValue");
        return inner.get().toStringValue();
    }

    private void throwIfEmptyOnConversion(String type) throws UdfException {
        if (!inner.isPresent()) {
            throw new UdfException("Invalid call. An empty optional cannot be converted to" + type + ".");
        }
    }

    public static OptionalValue empty(DataType dataType) throws UdfException {
        return new OptionalValue(Optional.empty(), DataType.ltOptional(dataType));
    }

    public static OptionalValue of(Value value) throws UdfException {
        if (value.getClass() == OptionalValue.class) {
            throw new UdfException("Invalid arguments. Creating an optional value of optional value is not allowed.");
        }
        return new OptionalValue(Optional.of(value), DataType.ltOptional(value.getDataType()));
    }
}
