/*
 * Copyright 2017-2022 Lenses.io Ltd
 */
package io.lenses.sql.udf.value;

import io.lenses.sql.udf.UdfException;
import io.lenses.sql.udf.datatype.DataType;

import java.math.BigDecimal;
import java.math.BigInteger;

/**
 * The date logical type represents a time, with no reference to a particular calendar, time zone or date,
 * with a precision of one day.
 *
 * A date stores the number of days after 01-01-1970.
 */
public class DateValue extends Primitive<Integer> {

    public DateValue(int value) {
        super(value, DataType.ltDate());
    }

    @Override
    public BigDecimalValue toBigDecimalValue() throws UdfException {
        TryConversion<BigDecimalValue> c = () -> new BigDecimalValue(new BigDecimal(underlying));
        return convert(c, DataType.ltBigDecimal());
    }

    @Override
    public BigIntValue toBigIntValue() throws UdfException {
        TryConversion<BigIntValue> c = () -> new BigIntValue(BigInteger.valueOf(underlying));
        return convert(c, DataType.ltBigInt());
    }

    @Override
    public BooleanValue toBooleanValue() throws UdfException {
        throw UdfException.conversionNotSupported(dataType, DataType.ltBoolean());
    }

    @Override
    public ByteValue toByteValue() throws UdfException {
        TryConversion<ByteValue> c = () -> new ByteValue(underlying.byteValue());
        return convert(c, DataType.ltByte());
    }

    @Override
    public DoubleValue toDoubleValue() throws UdfException {
        TryConversion<DoubleValue> c = () -> new DoubleValue(underlying.doubleValue());
        return convert(c, DataType.ltDouble());
    }

    @Override
    public FloatValue toFloatValue() throws UdfException {
        TryConversion<FloatValue> c = () -> new FloatValue(underlying.floatValue());
        return convert(c, DataType.ltFloat());
    }

    @Override
    public LongValue toLongValue()    throws UdfException {
        TryConversion<LongValue> c = () -> new LongValue(underlying.longValue());
        return convert(c, DataType.ltLong());
    }

    @Override
    public IntValue toIntValue() { return new IntValue(underlying); }

    @Override
    public StringValue toStringValue() throws UdfException {
        TryConversion<StringValue> c = () -> new StringValue(underlying.toString());
        return convert(c, DataType.ltString());
    }

}
