/*
 * Copyright 2017-2022 Lenses.io Ltd
 */
package io.lenses.sql.udf;

import io.lenses.sql.udf.datatype.DataType;
import io.lenses.sql.udf.value.Value;

import java.util.ArrayList;
import java.util.List;

public class UdfException extends Exception {

    public UdfException(String message) {
        super(message);
    }

    /**
     * Creates a UdfException indicating, that the conversion between two value types is not supported.
     * @param from The original value type.
     * @param to The target value type.
     * @return A UdfException with a description of the error.
     */
    public static UdfException conversionNotSupported(DataType from, DataType to) {
        return new UdfException("Conversion of value of type " + from.name + " to value of type " + to.name + " is not supported.");
    }

    /**
     * Creates a UdfException indicating, that the conversion between a container type to a primitive type is not supported.
     * @param from The container value type.
     * @param to The target primitive value type.
     * @return A UdfException with a description of the error.
     */
    public static UdfException containerToPrimitiveConversion(DataType from, DataType to) {
        return new UdfException("Conversion of container type " + from.name + " to primitive value of type " + to.name + " is not supported.");
    }

    /**
     * Creates a UdfException indicating, that the conversion between two value types is supported but failed.
     * @param from The original value type.
     * @param to The target value type.
     * @param value The underlying value.
     * @return A UdfException with a description of the error.
     */
    public static UdfException conversionError(DataType from, DataType to, Object value) {
        return new UdfException("Conversion of value of type " +
                from.name + "to value of type " +
                to.name + " failed for value: " +
                value);
    }

    /**
     * Creates a UdfException indicating that an invalid data type was encountered.
     *
     * @param functionName The name of the UDF.
     * @param position     The position of the invalid data type.
     * @param dataType     The invalid data type.
     * @return A UdfException with a description of the error.
     */
    public static UdfException argumentTypeNotSupported(String functionName,
                                                        int position,
                                                        DataType dataType) {
        return new UdfException("Function " + functionName +
                " does not support argument of type " + dataType.name +
                " at position " + position);
    }

    /**
     * Creates a UdfException indicating that an invalid combination of data types was encountered.
     *
     * @param functionName The name of the UDF.
     * @param dataTypes    List of input data types.
     * @return A UdfException with a description of the error.
     */
    public static UdfException argumentTypesNotSupported(String functionName,
                                                         List<DataType> dataTypes) {
        List<String> dtNames = new ArrayList<>();
        dataTypes.forEach((dt) -> dtNames.add(dt.name));
        return new UdfException("Function " + functionName +
                " does not support arguments with types " + dtNames);
    }

    /**
     * Creates a UDFException indicating that access of a struct field has failed.
     *
     * @param fields The fields of the struct value.
     * @param pos The position of the accessed field.
     * @return A UdfException with a description of the error.
     */
    public static UdfException fieldAccessError(List<Value> fields, int pos) {
        return new UdfException("Error while trying to access field at position " + pos + " of fields " + fields);
    }

    /**
     * Creates a UdfException indicating that the evaluation of the UDF for the given arguments failed.
     *
     * @param functionName The name of the UDF.
     * @param args         List of args passed to the evaluation method.
     * @param reason       The reason for the failure.
     * @return A UdfException with a description of the error.
     */
    public static UdfException evaluationFailed(String functionName,
                                                List<Value> args,
                                                String reason) {
        return new UdfException("Evaluation of function " + functionName +
                " for arguments " + args +
                " failed: " + reason);
    }

}
