/*
 * Copyright 2017-2022 Lenses.io Ltd
 */
package io.lenses.sql.udf.value;

import io.lenses.sql.udf.UdfException;
import io.lenses.sql.udf.datatype.DataType;

public abstract class Value {

    protected DataType dataType;

    protected <T extends Value> T convert(TryConversion<T> func, DataType toDataType) throws UdfException {
        try {
            return func.attemptConversion();
        } catch (Exception e) {
            throw UdfException.conversionError(dataType, toDataType, get());
        }
    }

    public Value(DataType dataType) {
        this.dataType = dataType;
    }

    public abstract Object get();

    public abstract Primitive asPrimitive() throws UdfException;

    public abstract Container asContainer() throws UdfException;

    public abstract RepeatedValue asRepeatedValue() throws UdfException;

    public abstract StructValue asStructValue() throws UdfException;

    public abstract boolean isPrimitive();

    public boolean isContainer() {
        return !isPrimitive();
    }

    public DataType getDataType() {
        return dataType;
    }

    /**
     * Returns this value converted to a NullValue.
     *
     * @return Discards this values underlying value and returns a NullValue.
     */
    public OptionalValue toOptional() throws UdfException {
        return OptionalValue.of(this);
    }

    /**
     * Returns this value converted to a BigDecimalValue or throws an exception.
     *
     * @return this Value converted to a BigDecimalValue
     * @throws UdfException if the conversion is not possible
     */
    public abstract BigDecimalValue toBigDecimalValue() throws UdfException;

    /**
     * Returns this value converted to a BigIntValue or throws an exception.
     *
     * @return this Value converted to a BigIntValue
     * @throws UdfException if the conversion is not possible
     */
    public abstract BigIntValue toBigIntValue() throws UdfException;

    /**
     * Returns this value converted to a BooleanValue or throws an exception.
     *
     * @return this Value converted to a BooleanValue
     * @throws UdfException if the conversion is not possible
     */
    public abstract BooleanValue toBooleanValue() throws UdfException;

    /**
     * Returns this value converted to a ByteValue or throws an exception.
     *
     * @return this Value converted to a ByteValue
     * @throws UdfException if the conversion is not possible
     */
    public abstract ByteValue toByteValue() throws UdfException;

    /**
     * Returns this value converted to a DoubleValue or throws an exception.
     *
     * @return this Value converted to a DoubleValue
     * @throws UdfException if the conversion is not possible
     */
    public abstract DoubleValue toDoubleValue() throws UdfException;

    /**
     * Returns this value converted to a FloatValue or throws an exception.
     *
     * @return this Value converted to a FloatValue
     * @throws UdfException if the conversion is not possible
     */
    public abstract FloatValue toFloatValue() throws UdfException;

    /**
     * Returns this value converted to a IntValue or throws an exception.
     *
     * @return this Value converted to a IntValue
     * @throws UdfException if the conversion is not possible
     */
    public abstract IntValue toIntValue() throws UdfException;

    /**
     * Returns this value converted to a LongValue or throws an exception.
     *
     * @return this Value converted to a LongValue
     * @throws UdfException if the conversion is not possible
     */
    public abstract LongValue toLongValue() throws UdfException;

    /**
     * Returns this value converted to a StringValue or throws an exception.
     *
     * @return this Value converted to a StringValue
     * @throws UdfException if the conversion is not possible
     */
    public abstract StringValue toStringValue() throws UdfException;


}

