/*
 * Decompiled with CFR 0.152.
 */
package io.lakefs;

import io.lakefs.BulkDeleter;
import io.lakefs.Constants;
import io.lakefs.FSConfiguration;
import io.lakefs.LakeFSClient;
import io.lakefs.LakeFSFileStatus;
import io.lakefs.clients.sdk.ApiException;
import io.lakefs.clients.sdk.BranchesApi;
import io.lakefs.clients.sdk.ObjectsApi;
import io.lakefs.clients.sdk.RepositoriesApi;
import io.lakefs.clients.sdk.model.ObjectCopyCreation;
import io.lakefs.clients.sdk.model.ObjectErrorList;
import io.lakefs.clients.sdk.model.ObjectStats;
import io.lakefs.clients.sdk.model.ObjectStatsList;
import io.lakefs.clients.sdk.model.Pagination;
import io.lakefs.clients.sdk.model.PathList;
import io.lakefs.clients.sdk.model.Repository;
import io.lakefs.clients.sdk.model.StorageConfig;
import io.lakefs.storage.CreateOutputStreamParams;
import io.lakefs.storage.PhysicalAddressTranslator;
import io.lakefs.storage.PresignedStorageAccessStrategy;
import io.lakefs.storage.SimpleStorageAccessStrategy;
import io.lakefs.storage.StorageAccessStrategy;
import io.lakefs.utils.ObjectLocation;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.AccessDeniedException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LakeFSFileSystem
extends FileSystem {
    public static final Logger LOG = LoggerFactory.getLogger(LakeFSFileSystem.class);
    public static final Logger OPERATIONS_LOG = LoggerFactory.getLogger((String)(LakeFSFileSystem.class + "[OPERATION]"));
    public static final String LAKEFS_DELETE_BULK_SIZE = "fs.lakefs.delete.bulk_size";
    private Configuration conf;
    private URI uri;
    private Path workingDirectory = new Path("/");
    private ClientFactory clientFactory;
    private LakeFSClient lfsClient;
    private int listAmount;
    private FileSystem fsForConfig;
    private boolean failedFSForConfig = false;
    private PhysicalAddressTranslator physicalAddressTranslator;
    private StorageAccessStrategy storageAccessStrategy;
    private Constants.AccessMode accessMode;
    private static File emptyFile = new File("/dev/null");
    private ExecutorService deleteExecutor = Executors.newSingleThreadExecutor(new ThreadFactory(){

        @Override
        public Thread newThread(Runnable r) {
            Thread t = new Thread(r);
            t.setDaemon(true);
            return t;
        }
    });

    public URI getUri() {
        return this.uri;
    }

    public void initialize(final URI name, final Configuration conf) throws IOException {
        this.initializeWithClientFactory(name, conf, new ClientFactory(){

            @Override
            public LakeFSClient newClient() throws IOException {
                return new LakeFSClient(name.getScheme(), conf);
            }
        });
    }

    void initializeWithClientFactory(URI name, Configuration conf, ClientFactory clientFactory) throws IOException {
        super.initialize(name, conf);
        this.uri = name;
        this.conf = conf;
        this.clientFactory = clientFactory;
        this.lfsClient = clientFactory.newClient();
        String host = name.getHost();
        if (host == null) {
            throw new IOException("Invalid repository specified");
        }
        this.setConf(conf);
        this.listAmount = FSConfiguration.getInt(conf, this.uri.getScheme(), "list.amount", 1000);
        String accessModeConf = FSConfiguration.get(conf, this.uri.getScheme(), "access.mode");
        this.accessMode = Constants.AccessMode.valueOf(((String)StringUtils.defaultIfBlank((CharSequence)accessModeConf, (CharSequence)Constants.AccessMode.SIMPLE.toString())).toUpperCase());
        if (this.accessMode == Constants.AccessMode.PRESIGNED) {
            this.storageAccessStrategy = new PresignedStorageAccessStrategy(this, this.lfsClient);
        } else if (this.accessMode == Constants.AccessMode.SIMPLE) {
            try {
                StorageConfig storageConfig = this.lfsClient.getInternalApi().getStorageConfig().execute();
                this.physicalAddressTranslator = new PhysicalAddressTranslator(storageConfig.getBlockstoreType(), storageConfig.getBlockstoreNamespaceValidityRegex());
            }
            catch (ApiException e) {
                throw new IOException("Failed to get lakeFS blockstore type", e);
            }
            this.storageAccessStrategy = new SimpleStorageAccessStrategy(this, this.lfsClient, conf, this.physicalAddressTranslator);
        } else {
            throw new IOException("Invalid access mode: " + (Object)((Object)this.accessMode));
        }
    }

    private synchronized FileSystem getFSForConfig() {
        if (this.fsForConfig != null) {
            return this.fsForConfig;
        }
        if (this.failedFSForConfig || this.accessMode == Constants.AccessMode.PRESIGNED || this.physicalAddressTranslator == null) {
            return null;
        }
        Path path = new Path(this.uri);
        ObjectLocation objectLoc = this.pathToObjectLocation(path);
        RepositoriesApi repositoriesApi = this.lfsClient.getRepositoriesApi();
        try {
            Repository repository = repositoriesApi.getRepository(objectLoc.getRepository()).execute();
            String storageNamespace = repository.getStorageNamespace();
            URI storageURI = URI.create(storageNamespace);
            Path physicalPath = this.physicalAddressTranslator.translate(storageNamespace);
            this.fsForConfig = physicalPath.getFileSystem(this.conf);
        }
        catch (ApiException | IOException | URISyntaxException e) {
            this.failedFSForConfig = true;
            LOG.warn("get underlying filesystem for {}: {} (use default values)", (Object)path, (Object)e);
        }
        return this.fsForConfig;
    }

    public long getDefaultBlockSize(Path path) {
        if (this.getFSForConfig() != null) {
            return this.getFSForConfig().getDefaultBlockSize(path);
        }
        return 0x2000000L;
    }

    public long getDefaultBlockSize() {
        if (this.getFSForConfig() != null) {
            return this.getFSForConfig().getDefaultBlockSize();
        }
        return 0x2000000L;
    }

    public FSDataInputStream open(Path path, int bufSize) throws IOException {
        OPERATIONS_LOG.trace("open({})", (Object)path);
        try {
            ObjectLocation objectLoc = this.pathToObjectLocation(path);
            return this.storageAccessStrategy.createDataInputStream(objectLoc, bufSize);
        }
        catch (ApiException e) {
            throw this.translateException("open: " + path, e);
        }
    }

    public RemoteIterator<LocatedFileStatus> listFiles(final Path f, final boolean recursive) throws FileNotFoundException, IOException {
        OPERATIONS_LOG.trace("list_files({}), recursive={}", (Object)f, (Object)recursive);
        return new RemoteIterator<LocatedFileStatus>(){
            private final ListingIterator iterator;
            {
                this.iterator = new ListingIterator(f, recursive, LakeFSFileSystem.this.listAmount);
            }

            public boolean hasNext() throws IOException {
                return this.iterator.hasNext();
            }

            public LocatedFileStatus next() throws IOException {
                LakeFSFileStatus status = this.iterator.next();
                BlockLocation[] locations = status.isFile() ? LakeFSFileSystem.this.getFileBlockLocations(status, 0L, status.getLen()) : new BlockLocation[]{};
                return new LocatedFileStatus((FileStatus)status, locations);
            }
        };
    }

    public FSDataOutputStream create(Path path, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        OPERATIONS_LOG.trace("create({})", (Object)path);
        try {
            LakeFSFileStatus status = this.getFileStatus(path);
            if (status.isDirectory()) {
                throw new FileAlreadyExistsException(path + " is a directory");
            }
            if (!overwrite) {
                throw new FileAlreadyExistsException(path + " already exists");
            }
        }
        catch (FileNotFoundException status) {
            // empty catch block
        }
        try {
            ObjectLocation objectLoc = this.pathToObjectLocation(path);
            return this.storageAccessStrategy.createDataOutputStream(objectLoc, new CreateOutputStreamParams().bufferSize(bufferSize).blockSize(blockSize).progress(progress));
        }
        catch (ApiException e) {
            throw new IOException("staging.getPhysicalAddress: " + e.getResponseBody(), e);
        }
    }

    public FSDataOutputStream append(Path path, int i, Progressable progressable) throws IOException {
        throw new UnsupportedOperationException("Append is not supported by LakeFSFileSystem");
    }

    public boolean rename(Path src, Path dst) throws IOException {
        LakeFSFileStatus srcStatus;
        OPERATIONS_LOG.trace("rename {} to {}", (Object)src, (Object)dst);
        ObjectLocation srcObjectLoc = this.pathToObjectLocation(src);
        ObjectLocation dstObjectLoc = this.pathToObjectLocation(dst);
        if (srcObjectLoc.getPath().isEmpty()) {
            LOG.error("rename: src {} is root directory", (Object)src);
            return false;
        }
        if (dstObjectLoc.getPath().isEmpty()) {
            LOG.error("rename: dst {} is root directory", (Object)dst);
            return false;
        }
        if (srcObjectLoc.equals(dstObjectLoc)) {
            LOG.debug("rename: src and dst refer to the same lakefs object location: {}", (Object)dst);
            return true;
        }
        if (!srcObjectLoc.onSameBranch(dstObjectLoc)) {
            LOG.error("rename: src {} and dst {} are not on the same branch. rename outside this scope is unsupported by lakefs.", (Object)src, (Object)dst);
            return false;
        }
        try {
            srcStatus = this.getFileStatus(src);
        }
        catch (FileNotFoundException ignored) {
            return false;
        }
        boolean result = srcStatus.isDirectory() ? this.renameDirectory(src, dst) : this.renameFile(srcStatus, dst);
        if (!src.getParent().equals((Object)dst.getParent())) {
            this.deleteEmptyDirectoryMarkers(dst.getParent());
            this.createDirectoryMarkerIfNotExists(src.getParent());
        }
        return result;
    }

    private boolean renameDirectory(Path src, Path dst) throws IOException {
        block7: {
            try {
                LakeFSFileStatus dstFileStatus = this.getFileStatus(dst);
                if (!dstFileStatus.isDirectory()) {
                    LOG.debug("renameDirectory: rename src {} to dst {}: src is a directory and dst is a file", (Object)src, (Object)dst);
                    return false;
                }
                if (!dstFileStatus.isEmptyDirectory()) {
                    LOG.debug("renameDirectory: rename src {} to dst {}: dst is a non-empty directory.", (Object)src, (Object)dst);
                    return false;
                }
                this.deleteHelper(this.pathToObjectLocation(dst).toDirectory());
            }
            catch (FileNotFoundException e) {
                LOG.debug("renameDirectory: dst {} does not exist", (Object)dst);
                if (this.isDirectory(dst.getParent())) break block7;
                return false;
            }
        }
        ListingIterator iterator = new ListingIterator(src, true, this.listAmount);
        iterator.setRemoveDirectory(false);
        while (iterator.hasNext()) {
            LakeFSFileStatus fileStatus = iterator.next();
            Path objDst = this.buildObjPathOnNonExistingDestinationDir(fileStatus.getPath(), src, dst);
            try {
                this.renameObject(fileStatus, objDst);
            }
            catch (IOException e) {
                throw new IOException("renameDirectory: failed to rename src directory " + src, e);
            }
        }
        return true;
    }

    private Path buildObjPathOnNonExistingDestinationDir(Path renamedObj, Path srcDir, Path dstDir) {
        String renamedPath = renamedObj.toUri().getPath();
        String srcPath = srcDir.toUri().getPath();
        if (srcPath.length() == renamedPath.length()) {
            return new Path(dstDir.toUri());
        }
        String renamedObjName = renamedPath.substring(srcPath.length() + 1);
        String newObjPath = dstDir.toUri() + "/" + renamedObjName;
        return new Path(newObjPath);
    }

    private Path buildObjPathOnExistingDestinationDir(Path srcObj, Path dstDir) {
        Path srcParent = srcObj.getParent();
        String filename = srcObj.toString().substring(srcParent.toString().length() + "/".length());
        return new Path(dstDir + "/" + filename);
    }

    private boolean renameFile(LakeFSFileStatus srcStatus, Path dst) throws IOException {
        block3: {
            try {
                LakeFSFileStatus dstFileStatus = this.getFileStatus(dst);
                LOG.debug("renameFile: dst {} exists and is a {}", (Object)dst, (Object)(dstFileStatus.isDirectory() ? "directory" : "file"));
                if (dstFileStatus.isDirectory()) {
                    dst = this.buildObjPathOnExistingDestinationDir(srcStatus.getPath(), dst);
                    LOG.debug("renameFile: use {} to create dst {}", (Object)srcStatus.getPath(), (Object)dst);
                }
            }
            catch (FileNotFoundException e) {
                LOG.debug("renameFile: dst does not exist, renaming src {} to a file called dst {}", (Object)srcStatus.getPath(), (Object)dst);
                if (this.isDirectory(dst.getParent())) break block3;
                return false;
            }
        }
        return this.renameObject(srcStatus, dst);
    }

    private boolean renameObject(LakeFSFileStatus srcStatus, Path dst) throws IOException {
        ObjectLocation srcObjectLoc = this.pathToObjectLocation(srcStatus.getPath());
        ObjectLocation dstObjectLoc = this.pathToObjectLocation(dst);
        if (srcStatus.isEmptyDirectory()) {
            srcObjectLoc = srcObjectLoc.toDirectory();
            dstObjectLoc = dstObjectLoc.toDirectory();
        }
        ObjectsApi objects = this.lfsClient.getObjectsApi();
        try {
            ObjectCopyCreation creationReq = new ObjectCopyCreation().srcRef(srcObjectLoc.getRef()).srcPath(srcObjectLoc.getPath());
            objects.copyObject(dstObjectLoc.getRepository(), dstObjectLoc.getRef(), dstObjectLoc.getPath(), creationReq).execute();
        }
        catch (ApiException e) {
            throw this.translateException("renameObject: src:" + srcStatus.getPath() + ", dst: " + dst + ", call to copyObject failed", e);
        }
        try {
            objects.deleteObject(srcObjectLoc.getRepository(), srcObjectLoc.getRef(), srcObjectLoc.getPath()).execute();
        }
        catch (ApiException e) {
            throw this.translateException("renameObject: src:" + srcStatus.getPath() + ", dst: " + dst + ", failed to delete src", e);
        }
        return true;
    }

    private IOException translateException(String msg, ApiException e) {
        int code = e.getCode();
        switch (code) {
            case 404: {
                return (FileNotFoundException)new FileNotFoundException(msg).initCause(e);
            }
            case 403: {
                return (AccessDeniedException)new AccessDeniedException(msg).initCause(e);
            }
        }
        return new IOException(msg, e);
    }

    public boolean delete(Path path, boolean recursive) throws IOException {
        LakeFSFileStatus status;
        OPERATIONS_LOG.trace("delete({}), recursive={}", (Object)path, (Object)recursive);
        try {
            status = this.getFileStatus(path);
        }
        catch (FileNotFoundException ignored) {
            return false;
        }
        boolean deleted = true;
        ObjectLocation loc = this.pathToObjectLocation(path);
        if (status.isDirectory()) {
            if (!recursive && !status.isEmptyDirectory()) {
                throw new IOException("Path is a non-empty directory: " + path);
            }
            if (status.isEmptyDirectory()) {
                loc = loc.toDirectory();
                deleted = this.deleteHelper(loc);
            } else {
                ObjectLocation location = this.pathToObjectLocation(path);
                try (BulkDeleter deleter = this.newDeleter(location.getRepository(), location.getRef());){
                    ListingIterator iterator = new ListingIterator(path, true, this.listAmount);
                    iterator.setRemoveDirectory(false);
                    while (iterator.hasNext()) {
                        LakeFSFileStatus fileStatus = iterator.next();
                        ObjectLocation fileLoc = this.pathToObjectLocation(fileStatus.getPath());
                        if (fileStatus.isDirectory()) {
                            fileLoc = fileLoc.toDirectory();
                        }
                        deleter.add(fileLoc.getPath());
                    }
                }
                catch (BulkDeleter.DeleteFailuresException e) {
                    LOG.error("delete(%s, %b): %s", new Object[]{path, recursive, e.toString()});
                    deleted = false;
                }
            }
        } else {
            deleted = this.deleteHelper(loc);
        }
        this.createDirectoryMarkerIfNotExists(path.getParent());
        return deleted;
    }

    private BulkDeleter newDeleter(String repository, String branch) throws IOException {
        final ObjectsApi objectsApi = this.clientFactory.newClient().getObjectsApi();
        return new BulkDeleter(this.deleteExecutor, new BulkDeleter.Callback(){

            @Override
            public ObjectErrorList apply(String repository, String branch, PathList pathList) throws ApiException {
                return objectsApi.deleteObjects(repository, branch, pathList).execute();
            }
        }, repository, branch, this.conf.getInt(LAKEFS_DELETE_BULK_SIZE, 0));
    }

    private boolean deleteHelper(ObjectLocation loc) throws IOException {
        try {
            ObjectsApi objectsApi = this.lfsClient.getObjectsApi();
            objectsApi.deleteObject(loc.getRepository(), loc.getRef(), loc.getPath()).execute();
        }
        catch (ApiException e) {
            if (e.getCode() == 404) {
                LOG.error("Could not delete: {}, reason: {}", (Object)loc, (Object)e.getResponseBody());
                return false;
            }
            throw new IOException("deleteObject", e);
        }
        return true;
    }

    void deleteEmptyDirectoryMarkers(Path f) {
        while (true) {
            try {
                LakeFSFileStatus status;
                ObjectLocation objectLocation = this.pathToObjectLocation(f);
                if (!objectLocation.isValidPath() || !(status = this.getFileStatus(f)).isDirectory() || !status.isEmptyDirectory()) break;
                this.deleteHelper(objectLocation.toDirectory());
            }
            catch (IOException iOException) {
                // empty catch block
            }
            if (f.isRoot()) break;
            f = f.getParent();
        }
    }

    private void createDirectoryMarkerIfNotExists(Path f) throws IOException {
        ObjectLocation objectLocation = this.pathToObjectLocation(f).toDirectory();
        if (!objectLocation.isValidPath()) {
            LOG.warn("Cannot create directory marker for invalid path {}", (Object)f.toString());
            return;
        }
        try {
            ObjectsApi objects = this.lfsClient.getObjectsApi();
            objects.uploadObject(objectLocation.getRepository(), objectLocation.getRef(), objectLocation.getPath()).ifNoneMatch("*").content(emptyFile).execute();
        }
        catch (ApiException e) {
            if (e.getCode() == 412) {
                LOG.trace("createDirectoryMarkerIfNotExists: Ignore {} response, marker exists");
                return;
            }
            throw new IOException("createDirectoryMarkerIfNotExists", e);
        }
    }

    public FileStatus[] listStatus(Path path) throws FileNotFoundException, IOException {
        OPERATIONS_LOG.trace("list_status({})", (Object)path);
        ObjectLocation objectLoc = this.pathToObjectLocation(path);
        ObjectsApi objectsApi = this.lfsClient.getObjectsApi();
        try {
            ObjectStats objectStat = objectsApi.statObject(objectLoc.getRepository(), objectLoc.getRef(), objectLoc.getPath()).userMetadata(Boolean.valueOf(false)).presign(Boolean.valueOf(false)).execute();
            LakeFSFileStatus fileStatus = this.convertObjectStatsToFileStatus(objectLoc, objectStat);
            return new FileStatus[]{fileStatus};
        }
        catch (ApiException e) {
            block8: {
                if (e.getCode() != 404) {
                    throw new IOException("statObject", e);
                }
                ArrayList<LakeFSFileStatus> fileStatuses = new ArrayList<LakeFSFileStatus>();
                ListingIterator iterator = new ListingIterator(path, false, this.listAmount);
                while (iterator.hasNext()) {
                    LakeFSFileStatus fileStatus = iterator.next();
                    fileStatuses.add(fileStatus);
                }
                if (!fileStatuses.isEmpty()) {
                    return fileStatuses.toArray(new FileStatus[0]);
                }
                try {
                    ObjectStats objectStat = objectsApi.statObject(objectLoc.getRepository(), objectLoc.getRef(), objectLoc.getPath() + "/").userMetadata(Boolean.valueOf(false)).presign(Boolean.valueOf(false)).execute();
                    LakeFSFileStatus fileStatus = this.convertObjectStatsToFileStatus(objectLoc, objectStat);
                    if (fileStatus.isEmptyDirectory()) {
                        return new FileStatus[0];
                    }
                }
                catch (ApiException e2) {
                    if (e2.getCode() == 404) break block8;
                    throw new IOException("statObject", e2);
                }
            }
            throw new FileNotFoundException("No such file or directory: " + path);
        }
    }

    public void setWorkingDirectory(Path path) {
        this.workingDirectory = path;
    }

    public Path getWorkingDirectory() {
        return this.workingDirectory;
    }

    public boolean mkdirs(Path path, FsPermission fsPermission) throws IOException {
        OPERATIONS_LOG.trace("mkdirs({})", (Object)path);
        try {
            LakeFSFileStatus fileStatus = this.getFileStatus(path);
            if (fileStatus.isDirectory()) {
                return true;
            }
            throw new FileAlreadyExistsException("Path is a file: " + path);
        }
        catch (FileNotFoundException e) {
            ObjectLocation objectLocation = this.pathToObjectLocation(path);
            Path branchRoot = new Path(objectLocation.toRefString());
            Path currentPath = path;
            do {
                try {
                    LakeFSFileStatus fileStatus = this.getFileStatus(currentPath);
                    if (fileStatus.isFile()) {
                        throw new FileAlreadyExistsException(String.format("Can't make directory for path '%s' since it is a file.", currentPath));
                    }
                }
                catch (FileNotFoundException fileNotFoundException) {
                    // empty catch block
                }
            } while ((currentPath = currentPath.getParent()) != null && !currentPath.equals((Object)branchRoot));
            this.createDirectoryMarker(path);
            return true;
        }
    }

    private void createDirectoryMarker(Path path) throws IOException {
        try {
            ObjectLocation objectLoc = this.pathToObjectLocation(path).toDirectory();
            FSDataOutputStream out = this.storageAccessStrategy.createDataOutputStream(objectLoc, null);
            out.close();
        }
        catch (ApiException e) {
            throw new IOException("createDirectoryMarker: " + e.getResponseBody(), e);
        }
    }

    public LakeFSFileStatus getFileStatus(Path path) throws IOException {
        OPERATIONS_LOG.trace("get_file_status({})", (Object)path);
        ObjectLocation objectLoc = this.pathToObjectLocation(path);
        if (objectLoc.getPath().isEmpty()) {
            if (this.isBranchExists(objectLoc.getRepository(), objectLoc.getRef())) {
                return new LakeFSFileStatus.Builder(path).isdir(true).build();
            }
            throw new FileNotFoundException(path + " not found");
        }
        ObjectsApi objectsApi = this.lfsClient.getObjectsApi();
        try {
            ObjectStats os = objectsApi.statObject(objectLoc.getRepository(), objectLoc.getRef(), objectLoc.getPath()).userMetadata(Boolean.valueOf(false)).presign(Boolean.valueOf(false)).execute();
            return this.convertObjectStatsToFileStatus(objectLoc, os);
        }
        catch (ApiException e) {
            if (e.getCode() != 404) {
                throw new IOException("statObject", e);
            }
            try {
                ObjectStats objectStat = objectsApi.statObject(objectLoc.getRepository(), objectLoc.getRef(), objectLoc.getPath() + "/").userMetadata(Boolean.valueOf(false)).presign(Boolean.valueOf(false)).execute();
                return this.convertObjectStatsToFileStatus(objectLoc, objectStat);
            }
            catch (ApiException e2) {
                if (e2.getCode() != 404) {
                    throw new IOException("statObject", e2);
                }
                ListingIterator iterator = new ListingIterator(path, true, 1);
                iterator.setRemoveDirectory(false);
                if (iterator.hasNext()) {
                    Path filePath = new Path(objectLoc.toString());
                    return new LakeFSFileStatus.Builder(filePath).isdir(true).build();
                }
                throw new FileNotFoundException(path + " not found");
            }
        }
    }

    @Nonnull
    private LakeFSFileStatus convertObjectStatsToFileStatus(ObjectLocation objectLocation, ObjectStats objectStat) throws IOException {
        long length = 0L;
        Long sizeBytes = objectStat.getSizeBytes();
        if (sizeBytes != null) {
            length = sizeBytes;
        }
        long modificationTime = 0L;
        Long mtime = objectStat.getMtime();
        if (mtime != null) {
            modificationTime = TimeUnit.SECONDS.toMillis(mtime);
        }
        Path filePath = new Path(ObjectLocation.formatPath(objectLocation.getScheme(), objectLocation.getRepository(), objectLocation.getRef(), objectStat.getPath()));
        String physicalAddress = objectStat.getPhysicalAddress();
        boolean isDir = LakeFSFileSystem.isDirectory(objectStat);
        boolean isEmptyDirectory = isDir && objectStat.getPathType() == ObjectStats.PathTypeEnum.OBJECT;
        long blockSize = isDir ? 0L : this.getDefaultBlockSize();
        LakeFSFileStatus.Builder builder = new LakeFSFileStatus.Builder(filePath).length(length).isdir(isDir).isEmptyDirectory(isEmptyDirectory).blockSize(blockSize).mTime(modificationTime).checksum(objectStat.getChecksum()).physicalAddress(physicalAddress);
        return builder.build();
    }

    public String getScheme() {
        return this.uri.getScheme();
    }

    public boolean exists(Path path) throws IOException {
        List l;
        ObjectStatsList osl;
        OPERATIONS_LOG.trace("exists({})", (Object)path);
        ObjectLocation objectLoc = this.pathToObjectLocation(path);
        if (objectLoc.getPath().isEmpty()) {
            return this.isBranchExists(objectLoc.getRepository(), objectLoc.getRef());
        }
        ObjectsApi objects = this.lfsClient.getObjectsApi();
        String directoryPath = objectLoc.toDirectory().getPath();
        try {
            osl = objects.listObjects(objectLoc.getRepository(), objectLoc.getRef()).userMetadata(Boolean.valueOf(false)).presign(Boolean.valueOf(false)).amount(Integer.valueOf(5)).prefix(objectLoc.getPath()).execute();
            l = osl.getResults();
            if (l.isEmpty()) {
                return false;
            }
            ObjectStats first = (ObjectStats)l.get(0);
            if (first.getPath().equals(objectLoc.getPath())) {
                return true;
            }
            for (ObjectStats stat : l) {
                if (stat.getPath().startsWith(directoryPath)) {
                    return true;
                }
                if (stat.getPath().compareTo(directoryPath) <= 0) continue;
                return false;
            }
            if (!osl.getPagination().getHasMore().booleanValue()) {
                return false;
            }
        }
        catch (ApiException e) {
            if (e.getCode() == 404) {
                return false;
            }
            throw new IOException("exists", e);
        }
        try {
            osl = objects.listObjects(objectLoc.getRepository(), objectLoc.getRef()).userMetadata(Boolean.valueOf(false)).presign(Boolean.valueOf(false)).amount(Integer.valueOf(1)).prefix(directoryPath).execute();
            l = osl.getResults();
            return !l.isEmpty();
        }
        catch (ApiException e) {
            throw new IOException("exists", e);
        }
    }

    private boolean isBranchExists(String repository, String branch) throws IOException {
        try {
            BranchesApi branches = this.lfsClient.getBranchesApi();
            branches.getBranch(repository, branch).execute();
            return true;
        }
        catch (ApiException e) {
            if (e.getCode() != 404) {
                throw new IOException("getBranch", e);
            }
            return false;
        }
    }

    @Nonnull
    public ObjectLocation pathToObjectLocation(Path path) {
        return ObjectLocation.pathToObjectLocation(this.workingDirectory, path);
    }

    private static boolean isDirectory(ObjectStats stat) {
        return stat.getPath().endsWith("/") || stat.getPathType() == ObjectStats.PathTypeEnum.COMMON_PREFIX;
    }

    public FSDataOutputStream createNonRecursive(Path path, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        Path parent = path.getParent();
        if (parent != null && !this.getFileStatus(parent).isDirectory()) {
            throw new FileAlreadyExistsException("Not a directory: " + parent);
        }
        return this.create(path, permission, flags.contains(CreateFlag.OVERWRITE), bufferSize, replication, blockSize, progress);
    }

    class ListingIterator
    implements RemoteIterator<LakeFSFileStatus> {
        private final ObjectLocation objectLocation;
        private final String delimiter;
        private final int amount;
        private boolean removeDirectory;
        private String nextOffset;
        private boolean last;
        private List<ObjectStats> chunk;
        private int pos;

        public ListingIterator(Path path, boolean recursive, int amount) {
            this.removeDirectory = recursive;
            this.chunk = Collections.emptyList();
            this.objectLocation = LakeFSFileSystem.this.pathToObjectLocation(path).toDirectory();
            this.delimiter = recursive ? "" : "/";
            this.last = false;
            this.pos = 0;
            this.amount = amount == 0 ? 1000 : amount;
            this.nextOffset = "";
        }

        public boolean hasNext() throws IOException {
            if (!this.last && this.pos >= this.chunk.size()) {
                this.readNextChunk();
            }
            return this.pos < this.chunk.size();
        }

        private void readNextChunk() throws IOException {
            String listingPath = this.objectLocation.getPath();
            do {
                try {
                    ObjectsApi objectsApi = LakeFSFileSystem.this.lfsClient.getObjectsApi();
                    ObjectStatsList resp = objectsApi.listObjects(this.objectLocation.getRepository(), this.objectLocation.getRef()).userMetadata(Boolean.valueOf(false)).presign(Boolean.valueOf(false)).after(this.nextOffset).amount(Integer.valueOf(this.amount)).delimiter(this.delimiter).prefix(this.objectLocation.getPath()).execute();
                    this.chunk = resp.getResults();
                    this.pos = 0;
                    Pagination pagination = resp.getPagination();
                    this.nextOffset = pagination.getNextOffset();
                    if (!pagination.getHasMore().booleanValue()) {
                        this.last = true;
                    }
                }
                catch (ApiException e) {
                    throw new IOException("listObjects", e);
                }
                this.chunk = this.chunk.stream().filter(item -> {
                    if (this.removeDirectory && LakeFSFileSystem.isDirectory(item)) {
                        return false;
                    }
                    return !item.getPath().equals(listingPath);
                }).collect(Collectors.toList());
            } while (this.chunk.isEmpty() && !this.last);
        }

        public boolean isRemoveDirectory() {
            return this.removeDirectory;
        }

        public void setRemoveDirectory(boolean removeDirectory) {
            this.removeDirectory = removeDirectory;
        }

        public LakeFSFileStatus next() throws IOException {
            if (!this.hasNext()) {
                throw new NoSuchElementException("No more entries");
            }
            ObjectStats objectStats = this.chunk.get(this.pos++);
            return LakeFSFileSystem.this.convertObjectStatsToFileStatus(this.objectLocation, objectStats);
        }
    }

    @FunctionalInterface
    private static interface BiFunctionWithIOException<U, V, R> {
        public R apply(U var1, V var2) throws IOException;
    }

    public static interface ClientFactory {
        public LakeFSClient newClient() throws IOException;
    }
}

