/*
 * Decompiled with CFR 0.152.
 */
package io.lakefs;

import io.lakefs.LakeFSFileStatus;
import io.lakefs.LakeFSFileSystem;
import io.lakefs.ObjectLocation;
import io.lakefs.StringUtils;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.util.Progressable;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemTracer
extends FileSystem {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemTracer.class);
    private static final String TRACER_WORKING_DIR = "fs.lakefs.tracer.working_dir";
    private static final String USE_LAKEFS_OUTPUT = "fs.lakefs.tracer.use_lakefs_output";
    private static final Object S3_URI_SCHEME = "s3";
    private static final Object RESULTS_COMPARISON = "[RESULTS_COMPARISON]";
    private boolean useLakeFSFileSystemResults;
    private LakeFSFileSystem lfsFileSystem;
    private FileSystem s3AFileSystem;
    private String s3aPathPrefix;
    private String lfsPathPrefix;

    private Path translateLakeFSPathToS3APath(Path path) {
        return this.replacePathPrefix(path, this.lfsPathPrefix, this.s3aPathPrefix);
    }

    private Path translateS3APathToLakeFSPath(Path path) {
        return this.replacePathPrefix(path, this.s3aPathPrefix, this.lfsPathPrefix);
    }

    private Path replacePathPrefix(Path path, String curPrefix, String newPrefix) {
        String p = path.toString();
        boolean isValidPath = p.startsWith(curPrefix);
        if (isValidPath) {
            String objRelativePath = StringUtils.trimLeadingSlash(p.substring(curPrefix.length()));
            String newPath = String.format("%s/%s", newPrefix, objRelativePath);
            LOG.trace("Converted {} to {}", (Object)path, (Object)newPath);
            return new Path(newPath);
        }
        LOG.error("Invalid path {}", (Object)path);
        return null;
    }

    public void initialize(URI name, Configuration conf) throws IOException {
        this.lfsFileSystem = new LakeFSFileSystem();
        this.lfsFileSystem.initialize(name, conf);
        ObjectLocation loc = this.lfsFileSystem.pathToObjectLocation(new Path(name));
        this.lfsPathPrefix = ObjectLocation.formatPath(loc.getScheme(), loc.getRepository(), loc.getRef());
        String tracerWorkingDir = conf.get(TRACER_WORKING_DIR);
        if (tracerWorkingDir == null) {
            throw new IOException("tracerWorkingDir is null");
        }
        this.s3aPathPrefix = String.format("%s://%s", S3_URI_SCHEME, tracerWorkingDir);
        Path s3aPath = new Path(this.s3aPathPrefix);
        this.s3AFileSystem = s3aPath.getFileSystem(conf);
        this.useLakeFSFileSystemResults = conf.getBoolean(USE_LAKEFS_OUTPUT, false);
        LOG.trace("Initialization finished, fs.lakefs.tracer.use_lakefs_output: {}", (Object)this.useLakeFSFileSystemResults);
    }

    public URI getUri() {
        LOG.trace("getUri");
        URI lakefsRes = this.lfsFileSystem.getUri();
        URI s3aRes = this.s3AFileSystem.getUri();
        LOG.trace("{}[getUri] lakefs: {}, s3a: {}", new Object[]{RESULTS_COMPARISON, lakefsRes, s3aRes});
        if (this.useLakeFSFileSystemResults) {
            return lakefsRes;
        }
        return s3aRes;
    }

    public Path makeQualified(Path path) {
        LOG.trace("makeQualified");
        Path lakefsRes = this.lfsFileSystem.makeQualified(path);
        Path s3aRes = this.s3AFileSystem.makeQualified(this.translateLakeFSPathToS3APath(path));
        LOG.trace("{}[makeQualified] lakefs: {}, s3a: {}", new Object[]{RESULTS_COMPARISON, lakefsRes, s3aRes});
        if (this.useLakeFSFileSystemResults) {
            return lakefsRes;
        }
        return this.translateS3APathToLakeFSPath(s3aRes);
    }

    public FSDataInputStream open(Path f, int bufferSize) throws IOException {
        LOG.trace("open(Path {}, bufferSize {})", (Object)f, (Object)bufferSize);
        FSDataInputStream lakefsRes = null;
        FSDataInputStream s3aRes = null;
        IOException lakeFSException = null;
        IOException s3aException = null;
        Path s3aPath = this.translateLakeFSPathToS3APath(f);
        try {
            lakefsRes = this.lfsFileSystem.open(f, bufferSize);
        }
        catch (IOException e) {
            lakeFSException = e;
            LOG.error("[open] Can't open {} with lakeFSFileSystem, exception {}", (Object)f, (Object)e.getMessage());
        }
        try {
            s3aRes = this.s3AFileSystem.open(this.translateLakeFSPathToS3APath(f), bufferSize);
        }
        catch (IOException e) {
            s3aException = e;
            LOG.error("[open] Can't open {} with S3AFileSystem, exception {}", (Object)s3aPath, (Object)e.getMessage());
        }
        if (this.useLakeFSFileSystemResults && lakeFSException != null) {
            LOG.trace("[open] exception by lakeFSFileSystem");
            throw lakeFSException;
        }
        if (!this.useLakeFSFileSystemResults && s3aException != null) {
            LOG.trace("[open] exception by S3AFileSystem");
            throw s3aException;
        }
        LOG.trace("{}[open] lakefs: {}, s3a: {}", new Object[]{RESULTS_COMPARISON, lakefsRes, s3aRes});
        if (this.useLakeFSFileSystemResults) {
            return lakefsRes;
        }
        return s3aRes;
    }

    public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        LOG.trace("create(Path {}, permission {}, overwrite {}, bufferSize {}, replication {}, blockSize {}, progress {})", new Object[]{f, permission, overwrite, bufferSize, replication, blockSize, progress});
        FSDataOutputStream lakeFSStream = null;
        FSDataOutputStream s3aStream = null;
        IOException lakeFSException = null;
        IOException s3aException = null;
        Path s3aPath = this.translateLakeFSPathToS3APath(f);
        try {
            lakeFSStream = this.lfsFileSystem.create(f, permission, overwrite, bufferSize, replication, blockSize, progress);
        }
        catch (IOException e) {
            lakeFSException = e;
            LOG.error("[create] Can't create {} with lakeFSFileSystem, exception {}", (Object)f, (Object)e.getMessage());
        }
        try {
            s3aStream = this.s3AFileSystem.create(s3aPath, permission, overwrite, bufferSize, replication, blockSize, progress);
        }
        catch (IOException e) {
            s3aException = e;
            LOG.error("[create] Can't create {} with S3AFileSystem, exception {}", (Object)s3aPath, (Object)e.getMessage());
        }
        if (this.useLakeFSFileSystemResults && lakeFSException != null) {
            LOG.trace("[create] exception by lakeFSFileSystem");
            if (s3aStream != null) {
                s3aStream.close();
            }
            throw lakeFSException;
        }
        if (!this.useLakeFSFileSystemResults && s3aException != null) {
            LOG.trace("[create] exception by S3AFileSystem");
            if (lakeFSStream != null) {
                lakeFSStream.close();
            }
            throw s3aException;
        }
        LOG.trace("{}[create] lakefs: {}, s3a: {}", new Object[]{RESULTS_COMPARISON, lakeFSStream, s3aStream});
        TracerOutputTStream tOutputStream = new TracerOutputTStream(lakeFSStream, s3aStream);
        return new FSDataOutputStream((OutputStream)tOutputStream, null);
    }

    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        LOG.trace("append(f {}, bufferSize {}, progress {})", new Object[]{f, bufferSize, progress});
        FSDataOutputStream lakefsRes = this.lfsFileSystem.append(f, bufferSize, progress);
        FSDataOutputStream s3aRes = this.s3AFileSystem.append(this.translateLakeFSPathToS3APath(f), bufferSize, progress);
        LOG.trace("{}[append] lakefs: {}, s3a: {}", new Object[]{RESULTS_COMPARISON, lakefsRes, s3aRes});
        if (this.useLakeFSFileSystemResults) {
            return lakefsRes;
        }
        return s3aRes;
    }

    public boolean rename(Path src, Path dst) throws IOException {
        LOG.trace("rename(src {}, dst {})", (Object)src, (Object)dst);
        boolean lakefsRes = false;
        boolean s3aRes = false;
        IOException lakeFSException = null;
        IOException s3aException = null;
        try {
            lakefsRes = this.lfsFileSystem.rename(src, dst);
        }
        catch (IOException e) {
            lakeFSException = e;
            LOG.error("[rename] Can't rename {} to {} with lakeFSFileSystem, exception {}", new Object[]{src, dst, e.getMessage()});
        }
        try {
            s3aRes = this.s3AFileSystem.rename(this.translateLakeFSPathToS3APath(src), this.translateLakeFSPathToS3APath(dst));
        }
        catch (IOException e) {
            s3aException = e;
            LOG.error("[rename] Can't rename {} to {} with S3AFileSystem, exception {}", new Object[]{src, dst, e.getMessage()});
        }
        if (this.useLakeFSFileSystemResults && lakeFSException != null) {
            LOG.trace("[rename] exception by lakeFSFileSystem");
            throw lakeFSException;
        }
        if (!this.useLakeFSFileSystemResults && s3aException != null) {
            LOG.trace("[rename] exception by S3AFileSystem");
            throw s3aException;
        }
        LOG.trace("{}[rename] lakefs: {}, s3a: {}", new Object[]{RESULTS_COMPARISON, lakefsRes, s3aRes});
        if (this.useLakeFSFileSystemResults) {
            return lakefsRes;
        }
        return s3aRes;
    }

    public boolean delete(Path f, boolean recursive) throws IOException {
        LOG.trace("delete(f {}, recursive {})", (Object)f, (Object)recursive);
        boolean lakefsRes = false;
        boolean s3aRes = false;
        IOException lakeFSException = null;
        IOException s3aException = null;
        try {
            lakefsRes = this.delete(f, recursive);
        }
        catch (IOException e) {
            lakeFSException = e;
            LOG.error("[delete] Can't delete {} with lakeFSFileSystem, exception {}", (Object)f, (Object)e.getMessage());
        }
        try {
            s3aRes = this.s3AFileSystem.delete(this.translateLakeFSPathToS3APath(f), recursive);
        }
        catch (IOException e) {
            s3aException = e;
            LOG.error("[delete] Can't delete {} to {} with S3AFileSystem, exception {}", (Object)f, (Object)e.getMessage());
        }
        if (this.useLakeFSFileSystemResults && lakeFSException != null) {
            LOG.trace("[delete] exception by lakeFSFileSystem");
            throw lakeFSException;
        }
        if (!this.useLakeFSFileSystemResults && s3aException != null) {
            LOG.trace("[delete] exception by S3AFileSystem");
            throw s3aException;
        }
        LOG.trace("{}[delete] lakefs: {}, s3a: {}", new Object[]{RESULTS_COMPARISON, lakefsRes, s3aRes});
        if (this.useLakeFSFileSystemResults) {
            return lakefsRes;
        }
        return s3aRes;
    }

    public FileStatus[] listStatus(Path f) throws FileNotFoundException, IOException {
        LOG.trace("listStatus(f {})", (Object)f);
        FileStatus[] lakefsRes = null;
        FileStatus[] s3aRes = null;
        IOException lakeFSException = null;
        IOException s3aException = null;
        Path s3aPath = this.translateLakeFSPathToS3APath(f);
        try {
            lakefsRes = this.lfsFileSystem.listStatus(f);
        }
        catch (IOException e) {
            lakeFSException = e;
            LOG.error("[listStatus] Can't list the status of {} with lakeFSFileSystem, exception {}", (Object)f, (Object)e.getMessage());
        }
        try {
            s3aRes = this.s3AFileSystem.listStatus(s3aPath);
        }
        catch (IOException e) {
            s3aException = e;
            LOG.error("[listStatus] Can't list the status of {} with S3AFileSystem, exception {}", (Object)s3aPath, (Object)e.getMessage());
        }
        if (this.useLakeFSFileSystemResults && lakeFSException != null) {
            LOG.trace("[listStatus] exception by lakeFSFileSystem");
            throw lakeFSException;
        }
        if (!this.useLakeFSFileSystemResults && s3aException != null) {
            LOG.trace("[listStatus] exception by S3AFileSystem");
            throw s3aException;
        }
        LOG.trace("{}[listStatus] lakefs: {}, s3a: {}", new Object[]{RESULTS_COMPARISON, lakefsRes, s3aRes});
        if (this.useLakeFSFileSystemResults) {
            return lakefsRes;
        }
        for (FileStatus stat : s3aRes) {
            Path filePath = stat.getPath();
            Path lfsPath = this.translateS3APathToLakeFSPath(filePath);
            stat.setPath(lfsPath);
        }
        return s3aRes;
    }

    public void setWorkingDirectory(Path newDir) {
        LOG.trace("setWorkingDirectory(new_dir {})", (Object)newDir);
        this.lfsFileSystem.setWorkingDirectory(newDir);
        this.s3AFileSystem.setWorkingDirectory(this.translateLakeFSPathToS3APath(newDir));
    }

    public Path getWorkingDirectory() {
        LOG.trace("getWorkingDirectory()");
        Path lakefsRes = this.lfsFileSystem.getWorkingDirectory();
        Path s3aRes = this.s3AFileSystem.getWorkingDirectory();
        LOG.trace("{}[getWorkingDirectory] lakefs: {}, s3a: {}", new Object[]{RESULTS_COMPARISON, lakefsRes, s3aRes});
        if (this.useLakeFSFileSystemResults) {
            return lakefsRes;
        }
        return s3aRes;
    }

    public boolean mkdirs(Path f, FsPermission permission) throws IOException {
        LOG.trace("mkdirs(f {}, permission {})", (Object)f, (Object)permission);
        boolean lakefsRes = false;
        boolean s3aRes = false;
        IOException lakeFSException = null;
        IOException s3aException = null;
        try {
            lakefsRes = this.lfsFileSystem.mkdirs(f, permission);
        }
        catch (IOException e) {
            lakeFSException = e;
            LOG.error("[mkdirs] Can't mkdir {} with lakeFSFileSystem, exception {}", (Object)f, (Object)e.getMessage());
        }
        try {
            s3aRes = this.s3AFileSystem.mkdirs(this.translateLakeFSPathToS3APath(f), permission);
        }
        catch (IOException e) {
            s3aException = e;
            LOG.error("[mkdirs] Can't mkdir {} to {} with S3AFileSystem, exception {}", (Object)f, (Object)e.getMessage());
        }
        if (this.useLakeFSFileSystemResults && lakeFSException != null) {
            LOG.trace("[mkdirs] exception by lakeFSFileSystem");
            throw lakeFSException;
        }
        if (!this.useLakeFSFileSystemResults && s3aException != null) {
            LOG.trace("[mkdirs] exception by S3AFileSystem");
            throw s3aException;
        }
        LOG.trace("{}[mkdirs] lakefs: {}, s3a: {}", new Object[]{RESULTS_COMPARISON, lakefsRes, s3aRes});
        if (this.useLakeFSFileSystemResults) {
            return lakefsRes;
        }
        return s3aRes;
    }

    public FileStatus getFileStatus(Path f) throws IOException {
        LOG.trace("getFileStatus(f {})", (Object)f);
        LakeFSFileStatus lakefsRes = null;
        FileStatus s3aRes = null;
        IOException lakeFSException = null;
        IOException s3aException = null;
        Path s3aPath = this.translateLakeFSPathToS3APath(f);
        try {
            lakefsRes = this.lfsFileSystem.getFileStatus(f);
        }
        catch (IOException e) {
            lakeFSException = e;
            LOG.error("[getFileStatus] Can't get {} file status with lakeFSFileSystem, exception {}", (Object)f, (Object)e.getMessage());
        }
        try {
            s3aRes = this.s3AFileSystem.getFileStatus(s3aPath);
        }
        catch (IOException e) {
            s3aException = e;
            LOG.error("[getFileStatus] Can't get {} file status with S3AFileSystem, exception {}", (Object)s3aPath, (Object)e.getMessage());
        }
        if (this.useLakeFSFileSystemResults && lakeFSException != null) {
            LOG.trace("[getFileStatus] exception by lakeFSFileSystem");
            throw lakeFSException;
        }
        if (!this.useLakeFSFileSystemResults && s3aException != null) {
            LOG.trace("[getFileStatus] exception by S3AFileSystem");
            throw s3aException;
        }
        LOG.trace("{}[getFileStatus] lakefs: {}, s3a: {}", new Object[]{RESULTS_COMPARISON, lakefsRes, s3aRes});
        if (this.useLakeFSFileSystemResults) {
            return lakefsRes;
        }
        Path lfsPath = this.translateS3APathToLakeFSPath(s3aPath);
        s3aRes.setPath(lfsPath);
        return s3aRes;
    }

    private class TracerOutputTStream
    extends OutputStream {
        private FSDataOutputStream lakeFSStream;
        private FSDataOutputStream s3aStream;

        public TracerOutputTStream(FSDataOutputStream lakeFSStream, FSDataOutputStream s3aStream) throws IOException {
            this.lakeFSStream = lakeFSStream;
            this.s3aStream = s3aStream;
        }

        @Override
        public void write(int b) throws IOException {
            if (this.lakeFSStream != null) {
                this.lakeFSStream.write(b);
            }
            if (this.s3aStream != null) {
                this.s3aStream.write(b);
            }
        }

        @Override
        public void write(@NotNull byte[] b) throws IOException {
            if (this.lakeFSStream != null) {
                this.lakeFSStream.write(b);
            }
            if (this.s3aStream != null) {
                this.s3aStream.write(b);
            }
        }

        @Override
        public void write(@NotNull byte[] b, int off, int len) throws IOException {
            if (this.lakeFSStream != null) {
                this.lakeFSStream.write(b, off, len);
            }
            if (this.s3aStream != null) {
                this.s3aStream.write(b, off, len);
            }
        }

        @Override
        public void flush() throws IOException {
            if (this.lakeFSStream != null) {
                this.lakeFSStream.flush();
            }
            if (this.s3aStream != null) {
                this.s3aStream.flush();
            }
        }

        @Override
        public void close() throws IOException {
            if (this.lakeFSStream != null) {
                this.lakeFSStream.close();
            }
            if (this.s3aStream != null) {
                this.s3aStream.close();
            }
        }
    }
}

