/*
 * Decompiled with CFR 0.152.
 */
package io.lakefs;

import com.amazonaws.services.s3.model.ObjectMetadata;
import io.lakefs.FSConfiguration;
import io.lakefs.LakeFSClient;
import io.lakefs.LakeFSFileStatus;
import io.lakefs.LakeFSLocatedFileStatus;
import io.lakefs.LinkOnCloseOutputStream;
import io.lakefs.MetadataClient;
import io.lakefs.ObjectLocation;
import io.lakefs.clients.api.ApiException;
import io.lakefs.clients.api.ObjectsApi;
import io.lakefs.clients.api.RepositoriesApi;
import io.lakefs.clients.api.StagingApi;
import io.lakefs.clients.api.model.ObjectStageCreation;
import io.lakefs.clients.api.model.ObjectStats;
import io.lakefs.clients.api.model.ObjectStatsList;
import io.lakefs.clients.api.model.Pagination;
import io.lakefs.clients.api.model.Repository;
import io.lakefs.clients.api.model.StagingLocation;
import io.lakefs.clients.api.model.StagingMetadata;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.AccessDeniedException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.util.Progressable;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LakeFSFileSystem
extends FileSystem {
    public static final Logger LOG = LoggerFactory.getLogger(LakeFSFileSystem.class);
    public static final Logger OPERATIONS_LOG = LoggerFactory.getLogger((String)(LakeFSFileSystem.class + "[OPERATION]"));
    private Configuration conf;
    private URI uri;
    private Path workingDirectory = new Path("/");
    private LakeFSClient lfsClient;
    private int listAmount;
    private FileSystem fsForConfig;

    private URI translateUri(URI uri) throws URISyntaxException {
        switch (uri.getScheme()) {
            case "s3": {
                return new URI("s3a", uri.getUserInfo(), uri.getHost(), uri.getPort(), uri.getPath(), uri.getQuery(), uri.getFragment());
            }
        }
        throw new RuntimeException(String.format("unsupported URI scheme %s", uri.getScheme()));
    }

    public URI getUri() {
        return this.uri;
    }

    public void initialize(URI name, Configuration conf) throws IOException {
        this.initializeWithClient(name, conf, new LakeFSClient(name.getScheme(), conf));
    }

    void initializeWithClient(URI name, Configuration conf, LakeFSClient lfsClient) throws IOException {
        super.initialize(name, conf);
        this.uri = name;
        this.conf = conf;
        this.lfsClient = lfsClient;
        String host = name.getHost();
        if (host == null) {
            throw new IOException("Invalid repository specified");
        }
        this.setConf(conf);
        this.listAmount = FSConfiguration.getInt(conf, this.uri.getScheme(), "list.amount", 1000);
        Path path = new Path(name);
        ObjectLocation objectLoc = this.pathToObjectLocation(path);
        RepositoriesApi repositoriesApi = lfsClient.getRepositories();
        try {
            Repository repository = repositoriesApi.getRepository(objectLoc.getRepository());
            String storageNamespace = repository.getStorageNamespace();
            URI storageURI = URI.create(storageNamespace);
            Path physicalPath = new Path(this.translateUri(storageURI));
            this.fsForConfig = physicalPath.getFileSystem(conf);
        }
        catch (ApiException | URISyntaxException e) {
            LOG.warn("get underlying filesystem for {}: {} (use default values)", (Object)path, (Object)e);
        }
    }

    protected <R> R withFileSystemAndTranslatedPhysicalPath(String physicalAddress, BiFunctionWithIOException<FileSystem, Path, R> f) throws URISyntaxException, IOException {
        URI uri = this.translateUri(new URI(physicalAddress));
        Path path = new Path(uri.toString());
        FileSystem fs = path.getFileSystem(this.conf);
        return f.apply(fs, path);
    }

    public long getDefaultBlockSize(Path path) {
        if (this.fsForConfig != null) {
            return this.fsForConfig.getDefaultBlockSize(path);
        }
        return 0x2000000L;
    }

    public long getDefaultBlockSize() {
        if (this.fsForConfig != null) {
            return this.fsForConfig.getDefaultBlockSize();
        }
        return 0x2000000L;
    }

    public FSDataInputStream open(Path path, int bufSize) throws IOException {
        OPERATIONS_LOG.trace("open({})", (Object)path);
        try {
            ObjectsApi objects = this.lfsClient.getObjects();
            ObjectLocation objectLoc = this.pathToObjectLocation(path);
            ObjectStats stats = objects.statObject(objectLoc.getRepository(), objectLoc.getRef(), objectLoc.getPath());
            return this.withFileSystemAndTranslatedPhysicalPath(stats.getPhysicalAddress(), (fs, p) -> fs.open(p, bufSize));
        }
        catch (ApiException e) {
            throw this.translateException("open: " + path, e);
        }
        catch (URISyntaxException e) {
            throw new IOException("open physical", e);
        }
    }

    public RemoteIterator<LocatedFileStatus> listFiles(Path f, boolean recursive) throws FileNotFoundException, IOException {
        OPERATIONS_LOG.trace("list_files({}), recursive={}", (Object)f, (Object)recursive);
        return LakeFSFileSystem.toLocatedFileStatusIterator(new ListingIterator(f, recursive, this.listAmount));
    }

    public FSDataOutputStream create(Path path, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        OPERATIONS_LOG.trace("create({})", (Object)path);
        try {
            ObjectLocation objectLoc = this.pathToObjectLocation(path);
            return this.createDataOutputStream((fs, fp) -> fs.create(fp, false, bufferSize, replication, blockSize, progress), objectLoc);
        }
        catch (ApiException e) {
            throw new IOException("staging.getPhysicalAddress: " + e.getResponseBody(), e);
        }
        catch (URISyntaxException e) {
            throw new IOException("underlying storage uri", e);
        }
    }

    @NotNull
    private FSDataOutputStream createDataOutputStream(BiFunctionWithIOException<FileSystem, Path, OutputStream> createStream, ObjectLocation objectLoc) throws ApiException, URISyntaxException, IOException {
        StagingApi staging = this.lfsClient.getStaging();
        StagingLocation stagingLoc = staging.getPhysicalAddress(objectLoc.getRepository(), objectLoc.getRef(), objectLoc.getPath());
        URI physicalUri = this.translateUri(new URI(Objects.requireNonNull(stagingLoc.getPhysicalAddress())));
        Path physicalPath = new Path(physicalUri.toString());
        FileSystem physicalFs = physicalPath.getFileSystem(this.conf);
        OutputStream physicalOut = createStream.apply(physicalFs, physicalPath);
        MetadataClient metadataClient = new MetadataClient(physicalFs);
        LinkOnCloseOutputStream out = new LinkOnCloseOutputStream(this, stagingLoc, objectLoc, physicalUri, metadataClient, physicalOut);
        return new FSDataOutputStream((OutputStream)out, null);
    }

    public FSDataOutputStream append(Path path, int i, Progressable progressable) throws IOException {
        throw new UnsupportedOperationException("Append is not supported by LakeFSFileSystem");
    }

    public boolean rename(Path src, Path dst) throws IOException {
        OPERATIONS_LOG.trace("rename {} to {}", (Object)src, (Object)dst);
        ObjectLocation srcObjectLoc = this.pathToObjectLocation(src);
        ObjectLocation dstObjectLoc = this.pathToObjectLocation(dst);
        if (srcObjectLoc.getPath().isEmpty()) {
            LOG.error("rename: src {} is root directory", (Object)src);
            return false;
        }
        if (dstObjectLoc.getPath().isEmpty()) {
            LOG.error("rename: dst {} is root directory", (Object)dst);
            return false;
        }
        if (srcObjectLoc.equals(dstObjectLoc)) {
            LOG.debug("rename: src and dst refer to the same lakefs object location: {}", (Object)dst);
            return true;
        }
        if (!srcObjectLoc.onSameBranch(dstObjectLoc)) {
            LOG.error("rename: src {} and dst {} are not on the same branch. rename outside this scope is unsupported by lakefs.", (Object)src, (Object)dst);
            return false;
        }
        LakeFSFileStatus srcStatus = this.getFileStatus(src);
        boolean result = srcStatus.isDirectory() ? this.renameDirectory(src, dst) : this.renameFile(srcStatus, dst);
        if (!src.getParent().equals((Object)dst.getParent())) {
            this.deleteEmptyDirectoryMarkers(dst.getParent());
            this.createDirectoryMarkerIfEmptyDirectory(src.getParent());
        }
        return result;
    }

    private boolean renameDirectory(Path src, Path dst) throws IOException {
        try {
            LakeFSFileStatus dstFileStatus = this.getFileStatus(dst);
            if (!dstFileStatus.isDirectory()) {
                throw new FileAlreadyExistsException("Failed rename " + src + " to " + dst + "; source is a directory and dest is a file");
            }
            if (!dstFileStatus.isEmptyDirectory()) {
                LOG.error("renameDirectory: rename src {} to dst {}: dst is a non-empty directory.", (Object)src, (Object)dst);
                return false;
            }
            this.deleteHelper(this.pathToObjectLocation(dst).toDirectory());
        }
        catch (FileNotFoundException e) {
            LOG.debug("renameDirectory: dst {} does not exist", (Object)dst);
        }
        ListingIterator iterator = new ListingIterator(src, true, this.listAmount);
        iterator.setRemoveDirectory(false);
        while (iterator.hasNext()) {
            LakeFSLocatedFileStatus locatedFileStatus = iterator.next();
            Path objDst = this.buildObjPathOnNonExistingDestinationDir(locatedFileStatus.getPath(), src, dst);
            try {
                this.renameObject(locatedFileStatus.toLakeFSFileStatus(), objDst);
            }
            catch (IOException e) {
                throw new IOException("renameDirectory: failed to rename src directory " + src, e);
            }
        }
        return true;
    }

    private Path buildObjPathOnNonExistingDestinationDir(Path renamedObj, Path srcDir, Path dstDir) {
        String renamedPath = renamedObj.toUri().getPath();
        String srcPath = srcDir.toUri().getPath();
        if (srcPath.length() == renamedPath.length()) {
            return new Path(dstDir.toUri());
        }
        String renamedObjName = renamedPath.substring(srcPath.length() + 1);
        String newObjPath = dstDir.toUri() + "/" + renamedObjName;
        return new Path(newObjPath);
    }

    private Path buildObjPathOnExistingDestinationDir(Path renamedObj, Path dstDir) {
        ObjectLocation renamedObjLoc = this.pathToObjectLocation(renamedObj);
        return new Path(dstDir + "/" + renamedObjLoc.getPath());
    }

    private boolean renameFile(LakeFSFileStatus srcStatus, Path dst) throws IOException {
        try {
            LakeFSFileStatus dstFileStatus = this.getFileStatus(dst);
            LOG.debug("renameFile: dst {} exists and is a {}", (Object)dst, (Object)(dstFileStatus.isDirectory() ? "directory" : "file"));
            if (!dstFileStatus.isDirectory()) {
                throw new FileAlreadyExistsException("Failed rename " + srcStatus.getPath() + " to " + dst + "; destination file already exists.");
            }
            dst = this.buildObjPathOnExistingDestinationDir(srcStatus.getPath(), dst);
        }
        catch (FileNotFoundException e) {
            LOG.debug("renameFile: dst does not exist, renaming src {} to a file called dst {}", (Object)srcStatus.getPath(), (Object)dst);
        }
        return this.renameObject(srcStatus, dst);
    }

    private boolean renameObject(LakeFSFileStatus srcStatus, Path dst) throws IOException {
        ObjectLocation srcObjectLoc = this.pathToObjectLocation(srcStatus.getPath());
        ObjectLocation dstObjectLoc = this.pathToObjectLocation(dst);
        if (srcStatus.isEmptyDirectory()) {
            srcObjectLoc = srcObjectLoc.toDirectory();
            dstObjectLoc = dstObjectLoc.toDirectory();
        }
        ObjectsApi objects = this.lfsClient.getObjects();
        ObjectStageCreation creationReq = new ObjectStageCreation().checksum(srcStatus.getChecksum()).sizeBytes(Long.valueOf(srcStatus.getLen())).physicalAddress(srcStatus.getPhysicalAddress());
        try {
            objects.stageObject(dstObjectLoc.getRepository(), dstObjectLoc.getRef(), dstObjectLoc.getPath(), creationReq);
        }
        catch (ApiException e) {
            throw this.translateException("renameObject: src:" + srcStatus.getPath() + ", dst: " + dst + ", failed to stage object", e);
        }
        try {
            objects.deleteObject(srcObjectLoc.getRepository(), srcObjectLoc.getRef(), srcObjectLoc.getPath());
        }
        catch (ApiException e) {
            throw this.translateException("renameObject: src:" + srcStatus.getPath() + ", dst: " + dst + ", failed to delete src", e);
        }
        return true;
    }

    private IOException translateException(String msg, ApiException e) {
        int code = e.getCode();
        switch (code) {
            case 404: {
                return (FileNotFoundException)new FileNotFoundException(msg).initCause(e);
            }
            case 403: {
                return (AccessDeniedException)new AccessDeniedException(msg).initCause(e);
            }
        }
        return new IOException(msg, e);
    }

    public boolean delete(Path path, boolean recursive) throws IOException {
        LakeFSFileStatus status;
        OPERATIONS_LOG.trace("delete({}), recursive={}", (Object)path, (Object)recursive);
        try {
            status = this.getFileStatus(path);
        }
        catch (FileNotFoundException ignored) {
            return false;
        }
        boolean deleted = true;
        ObjectLocation loc = this.pathToObjectLocation(path);
        if (status.isDirectory()) {
            if (!recursive && !status.isEmptyDirectory()) {
                throw new IOException("Path is a non-empty directory: " + path);
            }
            if (status.isEmptyDirectory()) {
                loc = loc.toDirectory();
                deleted = this.deleteHelper(loc);
            } else {
                ListingIterator iterator = new ListingIterator(path, true, this.listAmount);
                iterator.setRemoveDirectory(false);
                while (iterator.hasNext()) {
                    LakeFSLocatedFileStatus fileStatus = iterator.next();
                    ObjectLocation fileLoc = this.pathToObjectLocation(fileStatus.getPath());
                    if (fileStatus.isDirectory()) {
                        fileLoc = fileLoc.toDirectory();
                    }
                    this.deleteHelper(fileLoc);
                }
            }
        } else {
            deleted = this.deleteHelper(loc);
        }
        this.createDirectoryMarkerIfEmptyDirectory(path.getParent());
        return deleted;
    }

    private boolean deleteHelper(ObjectLocation loc) throws IOException {
        try {
            ObjectsApi objectsApi = this.lfsClient.getObjects();
            objectsApi.deleteObject(loc.getRepository(), loc.getRef(), loc.getPath());
        }
        catch (ApiException e) {
            if (e.getCode() == 404) {
                LOG.error("Could not delete: {}, reason: {}", (Object)loc, (Object)e.getResponseBody());
                return false;
            }
            throw new IOException("deleteObject", e);
        }
        return true;
    }

    void deleteEmptyDirectoryMarkers(Path f) {
        while (true) {
            try {
                ObjectLocation objectLocation = this.pathToObjectLocation(f);
                if (!objectLocation.isValidPath()) break;
                LakeFSFileStatus status = this.getFileStatus(f);
                if (status.isDirectory() && status.isEmptyDirectory()) {
                    this.deleteHelper(objectLocation.toDirectory());
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
            if (f.isRoot()) break;
            f = f.getParent();
        }
    }

    private void createDirectoryMarkerIfEmptyDirectory(Path f) throws IOException {
        ObjectLocation objectLocation = this.pathToObjectLocation(f);
        if (objectLocation.isValidPath() && !this.exists(f)) {
            this.createDirectoryMarker(f);
        }
    }

    public FileStatus[] listStatus(Path path) throws FileNotFoundException, IOException {
        OPERATIONS_LOG.trace("list_status({})", (Object)path);
        ObjectLocation objectLoc = this.pathToObjectLocation(path);
        try {
            ObjectsApi objectsApi = this.lfsClient.getObjects();
            ObjectStats objectStat = objectsApi.statObject(objectLoc.getRepository(), objectLoc.getRef(), objectLoc.getPath());
            LakeFSFileStatus fileStatus = this.convertObjectStatsToFileStatus(objectLoc, objectStat);
            return new FileStatus[]{fileStatus};
        }
        catch (ApiException e) {
            if (e.getCode() != 404) {
                throw new IOException("statObject", e);
            }
            ArrayList<LakeFSFileStatus> fileStatuses = new ArrayList<LakeFSFileStatus>();
            ListingIterator iterator = new ListingIterator(path, false, this.listAmount);
            while (iterator.hasNext()) {
                LakeFSLocatedFileStatus fileStatus = iterator.next();
                fileStatuses.add(fileStatus.toLakeFSFileStatus());
            }
            return fileStatuses.toArray(new FileStatus[0]);
        }
    }

    public void setWorkingDirectory(Path path) {
        this.workingDirectory = path;
    }

    public Path getWorkingDirectory() {
        return this.workingDirectory;
    }

    public boolean mkdirs(Path path, FsPermission fsPermission) throws IOException {
        OPERATIONS_LOG.trace("mkdirs({})", (Object)path);
        try {
            LakeFSFileStatus fileStatus = this.getFileStatus(path);
            if (fileStatus.isDirectory()) {
                return true;
            }
            throw new FileAlreadyExistsException("Path is a file: " + path);
        }
        catch (FileNotFoundException e) {
            ObjectLocation objectLocation = this.pathToObjectLocation(path);
            Path branchRoot = new Path(objectLocation.toRefString());
            Path currentPath = path;
            do {
                try {
                    LakeFSFileStatus fileStatus = this.getFileStatus(currentPath);
                    if (fileStatus.isFile()) {
                        throw new FileAlreadyExistsException(String.format("Can't make directory for path '%s' since it is a file.", currentPath));
                    }
                }
                catch (FileNotFoundException fileNotFoundException) {
                    // empty catch block
                }
            } while ((currentPath = currentPath.getParent()) != null && !currentPath.equals((Object)branchRoot));
            this.createDirectoryMarker(path);
            return true;
        }
    }

    private void createDirectoryMarker(Path path) throws IOException {
        try {
            ObjectLocation objectLoc = this.pathToObjectLocation(path).toDirectory();
            FSDataOutputStream out = this.createDataOutputStream(FileSystem::create, objectLoc);
            out.close();
        }
        catch (ApiException e) {
            throw new IOException("createDirectoryMarker: " + e.getResponseBody(), e);
        }
        catch (URISyntaxException e) {
            throw new IOException("createDirectoryMarker", e);
        }
    }

    void linkPhysicalAddress(ObjectLocation objectLoc, StagingLocation stagingLoc, URI physicalUri, MetadataClient metadataClient) throws IOException, ApiException {
        ObjectMetadata objectMetadata = metadataClient.getObjectMetadata(physicalUri);
        StagingMetadata metadata = new StagingMetadata().staging(stagingLoc).checksum(objectMetadata.getETag()).sizeBytes(Long.valueOf(objectMetadata.getContentLength()));
        StagingApi staging = this.lfsClient.getStaging();
        staging.linkPhysicalAddress(objectLoc.getRepository(), objectLoc.getRef(), objectLoc.getPath(), metadata);
    }

    public LakeFSFileStatus getFileStatus(Path path) throws IOException {
        OPERATIONS_LOG.trace("get_file_status({})", (Object)path);
        ObjectLocation objectLoc = this.pathToObjectLocation(path);
        ObjectsApi objectsApi = this.lfsClient.getObjects();
        try {
            ObjectStats objectStat = objectsApi.statObject(objectLoc.getRepository(), objectLoc.getRef(), objectLoc.getPath());
            return this.convertObjectStatsToFileStatus(objectLoc, objectStat);
        }
        catch (ApiException e) {
            if (e.getCode() != 404) {
                throw new IOException("statObject", e);
            }
            try {
                ObjectStats objectStat = objectsApi.statObject(objectLoc.getRepository(), objectLoc.getRef(), objectLoc.getPath() + "/");
                return this.convertObjectStatsToFileStatus(objectLoc, objectStat);
            }
            catch (ApiException e2) {
                if (e2.getCode() != 404) {
                    throw new IOException("statObject", e2);
                }
                ListingIterator iterator = new ListingIterator(path, true, 1);
                iterator.setRemoveDirectory(false);
                if (iterator.hasNext()) {
                    Path filePath = new Path(objectLoc.toString());
                    return new LakeFSFileStatus.Builder(filePath).isdir(true).build();
                }
                throw new FileNotFoundException(path + " not found");
            }
        }
    }

    @Nonnull
    private LakeFSFileStatus convertObjectStatsToFileStatus(ObjectLocation objectLocation, ObjectStats objectStat) throws IOException {
        try {
            long length = 0L;
            Long sizeBytes = objectStat.getSizeBytes();
            if (sizeBytes != null) {
                length = sizeBytes;
            }
            long modificationTime = 0L;
            Long mtime = objectStat.getMtime();
            if (mtime != null) {
                modificationTime = TimeUnit.SECONDS.toMillis(mtime);
            }
            Path filePath = new Path(ObjectLocation.formatPath(objectLocation.getScheme(), objectLocation.getRepository(), objectLocation.getRef(), objectStat.getPath()));
            String physicalAddress = objectStat.getPhysicalAddress();
            boolean isDir = LakeFSFileSystem.isDirectory(objectStat);
            boolean isEmptyDirectory = isDir && objectStat.getPathType() == ObjectStats.PathTypeEnum.OBJECT;
            long blockSize = isDir ? 0L : this.withFileSystemAndTranslatedPhysicalPath(physicalAddress, FileSystem::getDefaultBlockSize);
            LakeFSFileStatus.Builder builder = new LakeFSFileStatus.Builder(filePath).length(length).isdir(isDir).isEmptyDirectory(isEmptyDirectory).blockSize(blockSize).mTime(modificationTime).checksum(objectStat.getChecksum()).physicalAddress(physicalAddress);
            return builder.build();
        }
        catch (URISyntaxException e) {
            throw new IOException("uri", e);
        }
    }

    public String getScheme() {
        return this.uri.getScheme();
    }

    public FileStatus[] globStatus(Path pathPattern) throws IOException {
        FileStatus fStatus = new FileStatus(0L, false, 1, 20L, 1L, new Path("tal-test"));
        return new FileStatus[]{fStatus};
    }

    public boolean exists(Path path) throws IOException {
        OPERATIONS_LOG.trace("exists({})", (Object)path);
        ObjectsApi objects = this.lfsClient.getObjects();
        ObjectLocation objectLoc = this.pathToObjectLocation(path);
        try {
            objects.statObject(objectLoc.getRepository(), objectLoc.getRef(), objectLoc.getPath());
            return true;
        }
        catch (ApiException e) {
            if (e.getCode() != 404) {
                throw new IOException("statObject", e);
            }
            try {
                objects.statObject(objectLoc.getRepository(), objectLoc.getRef(), objectLoc.getPath() + "/");
                return true;
            }
            catch (ApiException e2) {
                if (e2.getCode() != 404) {
                    throw new IOException("statObject", e2);
                }
                ListingIterator iterator = new ListingIterator(path, true, 1);
                return iterator.hasNext();
            }
        }
    }

    @Nonnull
    public ObjectLocation pathToObjectLocation(Path path) {
        if (!path.isAbsolute()) {
            path = new Path(this.workingDirectory, path);
        }
        URI uri = path.toUri();
        ObjectLocation loc = new ObjectLocation();
        loc.setScheme(uri.getScheme());
        loc.setRepository(uri.getHost());
        String s = ObjectLocation.trimLeadingSlash(uri.getPath());
        int i = s.indexOf("/");
        if (i == -1) {
            loc.setRef(s);
            loc.setPath("");
        } else {
            loc.setRef(s.substring(0, i));
            loc.setPath(s.substring(i + 1));
        }
        return loc;
    }

    private LakeFSLocatedFileStatus toLakeFSLocatedFileStatus(LakeFSFileStatus status) throws IOException {
        BlockLocation[] blockLocations = status.isFile() ? this.getFileBlockLocations(status, 0L, status.getLen()) : null;
        return new LakeFSLocatedFileStatus(status, blockLocations);
    }

    private static boolean isDirectory(ObjectStats stat) {
        return stat.getPath().endsWith("/") || stat.getPathType() == ObjectStats.PathTypeEnum.COMMON_PREFIX;
    }

    public static RemoteIterator<LocatedFileStatus> toLocatedFileStatusIterator(RemoteIterator<? extends LocatedFileStatus> iterator) {
        return iterator;
    }

    class ListingIterator
    implements RemoteIterator<LakeFSLocatedFileStatus> {
        private final ObjectLocation objectLocation;
        private final String delimiter;
        private final int amount;
        private boolean removeDirectory;
        private String nextOffset;
        private boolean last;
        private List<ObjectStats> chunk;
        private int pos;

        public ListingIterator(Path path, boolean recursive, int amount) {
            this.removeDirectory = recursive;
            this.chunk = Collections.emptyList();
            this.objectLocation = LakeFSFileSystem.this.pathToObjectLocation(path).toDirectory();
            this.delimiter = recursive ? "" : "/";
            this.last = false;
            this.pos = 0;
            this.amount = amount == 0 ? 1000 : amount;
            this.nextOffset = "";
        }

        public boolean hasNext() throws IOException {
            if (!this.last && this.pos >= this.chunk.size()) {
                this.readNextChunk();
            }
            return this.pos < this.chunk.size();
        }

        private void readNextChunk() throws IOException {
            do {
                try {
                    ObjectsApi objectsApi = LakeFSFileSystem.this.lfsClient.getObjects();
                    ObjectStatsList resp = objectsApi.listObjects(this.objectLocation.getRepository(), this.objectLocation.getRef(), this.objectLocation.getPath(), this.nextOffset, Integer.valueOf(this.amount), this.delimiter);
                    this.chunk = resp.getResults();
                    this.pos = 0;
                    Pagination pagination = resp.getPagination();
                    this.nextOffset = pagination.getNextOffset();
                    if (!pagination.getHasMore().booleanValue()) {
                        this.last = true;
                    }
                }
                catch (ApiException e) {
                    throw new IOException("listObjects", e);
                }
                if (!this.removeDirectory) continue;
                this.chunk = this.chunk.stream().filter(item -> !LakeFSFileSystem.isDirectory(item)).collect(Collectors.toList());
            } while (this.chunk.isEmpty() && !this.last);
        }

        public boolean isRemoveDirectory() {
            return this.removeDirectory;
        }

        public void setRemoveDirectory(boolean removeDirectory) {
            this.removeDirectory = removeDirectory;
        }

        public LakeFSLocatedFileStatus next() throws IOException {
            if (!this.hasNext()) {
                throw new NoSuchElementException("No more entries");
            }
            ObjectStats objectStats = this.chunk.get(this.pos++);
            LakeFSFileStatus fileStatus = LakeFSFileSystem.this.convertObjectStatsToFileStatus(this.objectLocation, objectStats);
            return LakeFSFileSystem.this.toLakeFSLocatedFileStatus(fileStatus);
        }
    }

    @FunctionalInterface
    private static interface BiFunctionWithIOException<U, V, R> {
        public R apply(U var1, V var2) throws IOException;
    }
}

