package org.springframework.cloud.bootstrap.encrypt;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.bootstrap.encrypt.EncryptionBootstrapConfiguration.KeyCondition;
import org.springframework.cloud.bootstrap.encrypt.KeyProperties.KeyStore;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.security.crypto.encrypt.TextEncryptor;
import org.springframework.security.rsa.crypto.KeyStoreKeyFactory;
import org.springframework.security.rsa.crypto.RsaSecretEncryptor;

/**
 * @see org.springframework.cloud.context.encrypt.EncryptorFactory#create(String)
 */
@Configuration
@ConditionalOnClass({ TextEncryptor.class })
@AutoConfigureAfter(EncryptionBootstrapConfiguration.class)
@EnableConfigurationProperties(KeySpecProperties.class)
@ConditionalOnProperty(prefix = "key-spec", name = "enabled", matchIfMissing = true)
public class EncryptionBootstrapConfigurationAfter {

  @Configuration
  @Conditional(KeyCondition.class)
  @ConditionalOnClass(RsaSecretEncryptor.class)
  protected static class RsaEncryptionConfigurationCustom {
    @Autowired
    private KeyProperties key;
    @Autowired
    private KeySpecProperties keySpec;

    @Bean
    @Primary
    public TextEncryptor textEncryptor() {
      KeyStore keyStore = this.key.getKeyStore();
      if (keyStore.getLocation() != null && keyStore.getLocation().exists()) {
        return new RsaSecretEncryptor(new KeyStoreKeyFactory(keyStore.getLocation(), keyStore.getPassword().toCharArray()).getKeyPair(keyStore.getAlias(), keyStore.getSecret().toCharArray()),
            this.key.getRsa().getAlgorithm(), this.key.getRsa().getSalt(), this.key.getRsa().isStrong());
      }
      return this.keySpec.getTextEncryptor(this.key.getKey());
    }
  }

  @Configuration
  @Conditional(KeyCondition.class)
  @ConditionalOnMissingClass("org.springframework.security.rsa.crypto.RsaSecretEncryptor")
  protected static class VanillaEncryptionConfigurationCustom {
    @Autowired
    private KeyProperties key;
    @Autowired
    private KeySpecProperties keySpec;

    @Bean
    @Primary
    public TextEncryptor textEncryptor() {
      return this.keySpec.getTextEncryptor(this.key.getKey());
    }
  }
}
