package org.springframework.boot.autoconfigure.web;

import java.net.InetAddress;

import javax.servlet.Servlet;

import org.apache.catalina.connector.Connector;
import org.eclipse.jetty.server.NetworkTrafficServerConnector;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.AutoConfigureOrder;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.condition.SearchStrategy;
import org.springframework.boot.context.embedded.ConfigurableEmbeddedServletContainer;
import org.springframework.boot.context.embedded.EmbeddedServletContainerCustomizer;
import org.springframework.boot.context.embedded.EmbeddedServletContainerFactory;
import org.springframework.boot.context.embedded.jetty.JettyEmbeddedServletContainerFactory;
import org.springframework.boot.context.embedded.jetty.JettyServerCustomizer;
import org.springframework.boot.context.embedded.tomcat.TomcatEmbeddedServletContainerFactory;
import org.springframework.boot.context.embedded.undertow.UndertowBuilderCustomizer;
import org.springframework.boot.context.embedded.undertow.UndertowEmbeddedServletContainerFactory;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;

import io.undertow.Undertow.Builder;

@AutoConfigureOrder(Ordered.HIGHEST_PRECEDENCE)
@Configuration
@ConditionalOnClass({ Servlet.class })
@ConditionalOnWebApplication
@ConditionalOnProperty(prefix = "server.http", name = "port")
@ConfigurationProperties("server.http")
@AutoConfigureAfter(EmbeddedServletContainerAutoConfiguration.class)
public class EmbeddedServletContainerAutoConfigurationAfter {
  private int port = -1;
  private InetAddress address;

  @Bean
  @ConditionalOnBean(value = EmbeddedServletContainerFactory.class, search = SearchStrategy.CURRENT)
  public EmbeddedServletContainerCustomizer containerCustomizer() {
    return new EmbeddedServletContainerCustomizer() {
      @Override
      public void customize(ConfigurableEmbeddedServletContainer container) {
        if (container instanceof TomcatEmbeddedServletContainerFactory) {
          Connector connector = new Connector(TomcatEmbeddedServletContainerFactory.DEFAULT_PROTOCOL);
          connector.setPort(port);
          if (getAddress() != null) {
            connector.setDomain(getAddress().getHostAddress());
          }
          ((TomcatEmbeddedServletContainerFactory) container).addAdditionalTomcatConnectors(connector);
        }
        if (container instanceof JettyEmbeddedServletContainerFactory) {
          ((JettyEmbeddedServletContainerFactory) container).addServerCustomizers(new JettyServerCustomizer() {
            @Override
            public void customize(Server server) {
              ServerConnector serverConnector = new NetworkTrafficServerConnector(server);
              serverConnector.setPort(port);
              if (getAddress() != null) {
                serverConnector.setHost(getAddress().getHostAddress());
              }
              server.addConnector(serverConnector);
            }
          });
        }
        if (container instanceof UndertowEmbeddedServletContainerFactory) {
          ((UndertowEmbeddedServletContainerFactory) container).addBuilderCustomizers(new UndertowBuilderCustomizer() {
            @Override
            public void customize(Builder builder) {
              String host = "0.0.0.0";
              if (getAddress() != null) {
                host = getAddress().getHostAddress();
              }
              builder.addHttpListener(port, host);
            }
          });
        }

      }
    };
  }

  public InetAddress getAddress() {
    return address;
  }

  public void setAddress(InetAddress address) {
    this.address = address;
  }

  public int getPort() {
    return port;
  }

  public void setPort(int port) {
    this.port = port;
  }
}
