package org.springframework.boot.autoconfigure.transaction;

import java.util.Properties;

import org.springframework.aop.aspectj.AspectJExpressionPointcutAdvisor;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Role;
import org.springframework.data.repository.util.TxUtils;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.config.TransactionManagementConfigUtils;
import org.springframework.transaction.interceptor.NameMatchTransactionAttributeSource;
import org.springframework.transaction.interceptor.TransactionAttributeSource;
import org.springframework.transaction.interceptor.TransactionInterceptor;

@Configuration
@ConditionalOnClass(PlatformTransactionManager.class)
@AutoConfigureAfter({ TransactionAutoConfiguration.class })
public class TransactionAutoConfigurationAfter {
  /**
   * @see org.springframework.transaction.annotation.ProxyTransactionManagementConfiguration#transactionAdvisor()
   * @see org.springframework.aop.aspectj.AspectJExpressionPointcutAdvisor#setOrder(int)
   */
  @Bean(name = TransactionManagementConfigUtils.TRANSACTION_ADVISOR_BEAN_NAME)
  @Role(BeanDefinition.ROLE_INFRASTRUCTURE)
  @ConfigurationProperties("transaction.advisor")
  public AspectJExpressionPointcutAdvisor transactionAdvisor(@Qualifier("transactionInterceptor") TransactionInterceptor transactionInterceptor) {
    String expression = "execution(* repository..*Service.*(..)) || execution(* repository.jpa..*Repository.*(..)) || " // custom
        + "execution(* com.repository..*Service.*(..)) || execution(* com.repository..*Repository.*(..)) || " // custom
        + "execution(* org.springframework.data.repository..*Repository.*(..)) || execution(* org.springframework.data.jpa.repository.support..*Repository.*(..)) || " // data
        + "execution(* org.springframework.batch.core.repository..*Repository.*(..)) || execution(* org.springframework.batch.core.repository.support..*Repository.*(..))";

    AspectJExpressionPointcutAdvisor aspectJExpressionPointcutAdvisor = new AspectJExpressionPointcutAdvisor();
    aspectJExpressionPointcutAdvisor.setExpression(expression);
    aspectJExpressionPointcutAdvisor.setAdvice(transactionInterceptor);
    return aspectJExpressionPointcutAdvisor;
  }

  /**
   * @see org.springframework.transaction.annotation.ProxyTransactionManagementConfiguration#transactionAttributeSource()
   * @see org.springframework.transaction.interceptor.MatchAlwaysTransactionAttributeSource
   * @see org.springframework.transaction.annotation.ProxyTransactionManagementConfiguration#transactionInterceptor()
   * @see org.springframework.transaction.interceptor.TransactionProxyFactoryBean
   */
  @Bean
  @Role(BeanDefinition.ROLE_INFRASTRUCTURE)
  @ConfigurationProperties("transaction.interceptor")
  public TransactionInterceptor transactionInterceptor(@Qualifier(TxUtils.DEFAULT_TRANSACTION_MANAGER) PlatformTransactionManager transactionManager,
      TransactionAttributeSource... transactionAttributeSources) {
    TransactionInterceptor transactionInterceptor = new TransactionInterceptor();
    transactionInterceptor.setTransactionManager(transactionManager);
    transactionInterceptor.setTransactionAttributeSources(transactionAttributeSources);
    return transactionInterceptor;
  }

  /**
   * @see org.springframework.transaction.annotation.AnnotationTransactionAttributeSource
   */
  @Bean
  @Role(BeanDefinition.ROLE_INFRASTRUCTURE)
  @ConfigurationProperties("transaction.attribute-source.name-match")
  public NameMatchTransactionAttributeSource nameMatchTransactionAttributeSource() {
    Properties attribute = new Properties();
    attribute.put("count*", "PROPAGATION_REQUIRED,ISOLATION_READ_UNCOMMITTED,readOnly");
    attribute.put("get*", "PROPAGATION_REQUIRED,ISOLATION_READ_UNCOMMITTED,readOnly");
    attribute.put("find*", "PROPAGATION_REQUIRED,ISOLATION_READ_UNCOMMITTED,readOnly");
    attribute.put("exist*", "PROPAGATION_REQUIRED,ISOLATION_READ_UNCOMMITTED,readOnly");
    attribute.put("load*", "PROPAGATION_REQUIRED,ISOLATION_READ_UNCOMMITTED,readOnly");
    attribute.put("*", "PROPAGATION_REQUIRED,ISOLATION_READ_COMMITTED");

    NameMatchTransactionAttributeSource transactionAttributeSource = new NameMatchTransactionAttributeSource();
    transactionAttributeSource.setProperties(attribute);
    return transactionAttributeSource;
  }
}
