package org.springframework.boot.autoconfigure.security.oauth2.resource;

import java.security.KeyPair;

import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.Resource;
import org.springframework.security.oauth2.config.annotation.web.configuration.AuthorizationServerEndpointsConfiguration;
import org.springframework.security.oauth2.provider.token.store.JwtAccessTokenConverter;
import org.springframework.security.oauth2.provider.token.store.KeyStoreKeyFactory;
import org.springframework.util.StringUtils;

@Configuration
@ConditionalOnMissingBean(AuthorizationServerEndpointsConfiguration.class)
@AutoConfigureAfter(ResourceServerTokenServicesConfiguration.class)
public class ResourceServerTokenServicesConfigurationAfter {
  @Configuration
  @ConditionalOnClass(name = "org.springframework.security.oauth2.provider.token.store.KeyStoreKeyFactory")
  @ConditionalOnProperty(prefix = "security.oauth2.resource.jwt", name = { "key-store", "key-store-password", "key-alias" })
  @ConfigurationProperties(prefix = "security.oauth2.resource.jwt")
  protected static class JwtTokenServicesConfigurationCustom {
    private String keyAlias;
    private String keyPassword;
    private Resource keyStore;
    private String keyStorePassword;

    @Bean
    public JwtAccessTokenConverterConfigurer jwtAccessTokenConverterConfigurer() {
      return new JwtAccessTokenConverterConfigurer() {
        @Override
        public void configure(JwtAccessTokenConverter converter) {
          KeyStoreKeyFactory keyStoreKeyFactory = new KeyStoreKeyFactory(keyStore, keyStorePassword.toCharArray());
          KeyPair keyPair = StringUtils.hasText(keyPassword) ? keyStoreKeyFactory.getKeyPair(keyAlias, keyPassword.toCharArray()) : keyStoreKeyFactory.getKeyPair(keyAlias);
          converter.setKeyPair(keyPair);
        }
      };
    }

    public String getKeyAlias() {
      return keyAlias;
    }

    public void setKeyAlias(String keyAlias) {
      this.keyAlias = keyAlias;
    }

    public String getKeyPassword() {
      return keyPassword;
    }

    public void setKeyPassword(String keyPassword) {
      this.keyPassword = keyPassword;
    }

    public Resource getKeyStore() {
      return keyStore;
    }

    public void setKeyStore(Resource keyStore) {
      this.keyStore = keyStore;
    }

    public String getKeyStorePassword() {
      return keyStorePassword;
    }

    public void setKeyStorePassword(String keyStorePassword) {
      this.keyStorePassword = keyStorePassword;
    }
  }
}
