package org.springframework.boot.autoconfigure.mobile;

import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.template.TemplateAvailabilityProviders;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.mobile.device.view.LiteDeviceDelegatingViewResolver;
import org.springframework.web.servlet.ViewResolver;

@Configuration
@ConditionalOnWebApplication
@ConditionalOnClass(LiteDeviceDelegatingViewResolver.class)
@ConditionalOnProperty(prefix = "spring.mobile.devicedelegatingviewresolver", name = "enabled", havingValue = "true")
@AutoConfigureAfter(DeviceDelegatingViewResolverAutoConfiguration.class)
public class DeviceDelegatingViewResolverAutoConfigurationAfter {

  /**
   * @see org.springframework.boot.autoconfigure.mobile.DeviceDelegatingViewResolverAutoConfiguration.LiteDeviceDelegatingViewResolverFactoryConfiguration#deviceDelegatingViewResolverFactory(DeviceDelegatingViewResolverProperties)
   */
  @Configuration
  protected static class LiteDeviceDelegatingViewResolverFactoryConfigurationCustom {
    @Bean
    @Primary
    public DeviceDelegatingViewResolverFactory deviceDelegatingViewResolverFactoryCustom(DeviceDelegatingViewResolverProperties properties) {
      return new DeviceDelegatingViewResolverFactory(properties) {
        @Override
        public LiteDeviceDelegatingViewResolver createViewResolver(ViewResolver delegate, int delegatingOrder) {
          LiteDeviceDelegatingViewResolver resolver = new LiteDeviceDelegatingViewResolver(delegate) {
            private TemplateAvailabilityProviders templateAvailabilityProviders;

            @Override
            protected String getDeviceViewName(String viewName) {
              String deviceViewName = super.getDeviceViewName(viewName);
              return getEnableFallback() && this.templateAvailabilityProviders.getProvider(deviceViewName, getApplicationContext()) == null ? viewName : deviceViewName;
            }

            @Override
            protected void initApplicationContext(ApplicationContext context) {
              super.initApplicationContext(context);
              this.templateAvailabilityProviders = new TemplateAvailabilityProviders(context);
            }
          };
          resolver.setEnableFallback(properties.isEnableFallback());
          resolver.setNormalPrefix(properties.getNormalPrefix());
          resolver.setNormalSuffix(properties.getNormalSuffix());
          resolver.setMobilePrefix(properties.getMobilePrefix());
          resolver.setMobileSuffix(properties.getMobileSuffix());
          resolver.setTabletPrefix(properties.getTabletPrefix());
          resolver.setTabletSuffix(properties.getTabletSuffix());
          resolver.setOrder(delegatingOrder);
          return resolver;
        }
      };
    }
  }

}
