package org.springframework.boot.autoconfigure.jdbc;

import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;

import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.jdbc.datasource.embedded.EmbeddedDatabaseType;

/**
 * <code>
 &#64;SuppressWarnings("unchecked")
    private void properties(DataSourceBuilder builder, Map<String, String> properties) {
      if (CollectionUtils.isEmpty(properties)) {
        return;
      }
      Field field = ReflectionUtils.findField(builder.getClass(), "properties");
      if (field != null) {
        ReflectionUtils.makeAccessible(field);
        Object object = ReflectionUtils.getField(field, builder);
        if (object instanceof Map) {
          ((Map<Object, Object>) object).putAll(properties);
        }
      }
    }

    &#64;SuppressWarnings("unchecked")
    private DataSource dataSource(DataSourceBuilder builder, List<Class<? extends DataSource>> proxys, ClassLoader classLoader) {
      DataSource dataSource = builder.build();

      List<Class<? extends DataSource>> dataSourceProxys = proxys;
      if (CollectionUtils.isEmpty(dataSourceProxys)) {
        dataSourceProxys = new ArrayList<Class<? extends DataSource>>();
        for (String name : new String[] { "com.p6spy.engine.spy.P6DataSource", "net.ttddyy.dsproxy.support.ProxyDataSource", "com.vladmihalcea.flexypool.FlexyPoolDataSource",
            "org.springframework.jdbc.datasource.LazyConnectionDataSourceProxy" }) {
          try {
            dataSourceProxys.add((Class<? extends DataSource>) ClassUtils.forName(name, classLoader));
          }
          catch (Exception ex) {
            // ignore
          }
        }
      }
      for (Class<? extends DataSource> proxy : dataSourceProxys) {
        try {
          Constructor<? extends DataSource> constructor = ClassUtils.getConstructorIfAvailable(proxy, DataSource.class);
          if (constructor == null) {
            throw new IllegalStateException("No constructor accepting real DataSource in proxy class " + proxy);
          }
          dataSource = BeanUtils.instantiateClass(constructor, dataSource);
        }
        catch (RuntimeException e) {
          if (proxys.contains(proxy)) {
            throw e;
          }
          logger.warn("Can't wrap data source: " + dataSource + " in proxy data source of type: " + dataSource.getClass().getName() + " due to thrown exception: " + e + ". "
              + "Please consider fix exception in data source implementation" + " or explicitly set 'spring.datasource.proxyType' with" + " appropriate list of proxy data source providers.");
        }
      }
      return dataSource;
    }
 * </code>
 * 
 * @see org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor
 * @see org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration
 * @see org.springframework.boot.autoconfigure.flyway.FlywayAutoConfiguration#FlywayInitializerJpaDependencyConfiguration
 */
@Configuration
@ConditionalOnBean(DataSource.class)
@ConditionalOnClass({ DataSource.class, EmbeddedDatabaseType.class })
@AutoConfigureAfter(DataSourceAutoConfiguration.class)
public class DataSourceMultipleAutoConfiguration {
  @Bean
  @ConfigurationProperties(prefix = "spring.datasource")
  public DataSourceMultipleListener dataSourceMultipleListener() {
    return new DataSourceMultipleListener();
  }

  /**
   * @see org.springframework.context.ApplicationContextInitializer
   * @see org.springframework.boot.context.event.ApplicationReadyEvent
   * @see org.springframework.boot.context.event.ApplicationPreparedEvent
   * @see org.springframework.boot.context.event.ApplicationEnvironmentPreparedEvent
   */
  protected static class DataSourceMultipleListener implements ApplicationListener<ApplicationReadyEvent> {
    private List<DataSourceMultipleProperties> multiple = new ArrayList<DataSourceMultipleProperties>();

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
      for (DataSourceMultipleProperties dataSourceMultipleProperties : this.multiple) {
        event.getApplicationContext().getBeanFactory().registerSingleton(dataSourceMultipleProperties.getName(), dataSourceMultipleProperties.initializeDataSourceBuilder().build());
      }
    }

    public List<DataSourceMultipleProperties> getMultiple() {
      return multiple;
    }

    public void setMultiple(List<DataSourceMultipleProperties> multiple) {
      this.multiple = multiple;
    }
  }
}