package org.springframework.boot.autoconfigure.jdbc;

import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.context.ApplicationContext;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.AnnotationAwareOrderComparator;

public class DataSourceDecoratorPostProcessor implements BeanPostProcessor, Ordered {
  private int order = Ordered.LOWEST_PRECEDENCE;

  private final ApplicationContext applicationContext;
  private final DataSourceDecoratorProperties properties;

  DataSourceDecoratorPostProcessor(DataSourceDecoratorProperties properties, ApplicationContext applicationContext) {
    this.applicationContext = applicationContext;
    this.properties = properties;
  }

  @Override
  public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
    return bean;
  }

  /**
   * <code>
  new org.springframework.boot.bind.RelaxedDataBinder(dataSource).bind(new org.springframework.beans.MutablePropertyValues(this.properties.getProperties()));
  new org.springframework.boot.context.properties.bind.Binder(new org.springframework.boot.context.properties.source.MapConfigurationPropertySource(this.properties.getProperties())).bind(org.springframework.boot.context.properties.source.ConfigurationPropertyName.EMPTY, org.springframework.boot.context.properties.bind.Bindable.ofInstance(dataSource));
   * </code>
   */
  @Override
  public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
    if (bean instanceof DataSource && !this.properties.getExcludes().contains(beanName)) {
      DataSource dataSource = (DataSource) bean;
      new org.springframework.boot.bind.RelaxedDataBinder(dataSource).bind(new org.springframework.beans.MutablePropertyValues(this.properties.getProperties()));
      List<DataSourceDecorator> decorators = new ArrayList<DataSourceDecorator>(this.applicationContext.getBeansOfType(DataSourceDecorator.class).values());
      AnnotationAwareOrderComparator.sort(decorators);
      for (DataSourceDecorator decorator : decorators) {
        dataSource = decorator.decorate(dataSource);
      }
      return dataSource;
    }
    return bean;
  }

  @Override
  public int getOrder() {
    return this.order;
  }
}