package org.springframework.boot.autoconfigure.cache;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.cache.CacheManager;
import org.springframework.cache.annotation.CachingConfigurer;
import org.springframework.cache.interceptor.CacheAspectSupport;
import org.springframework.cache.interceptor.CacheErrorHandler;
import org.springframework.cache.interceptor.CacheResolver;
import org.springframework.cache.interceptor.KeyGenerator;
import org.springframework.cache.interceptor.KeyGeneratorImpl;
import org.springframework.cache.interceptor.SimpleCacheErrorHandler;
import org.springframework.cache.interceptor.SimpleCacheResolver;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * <code>
  &#64;Bean
  public KeyGenerator customKeyGenerator() {
    return new KeyGenerator() {
      &#64;Override
      public Object generate(Object o, Method method, Object... objects) {
        StringBuilder sb = new StringBuilder();
        sb.append(o.getClass().getName());
        sb.append(method.getName());
        for (Object obj : objects) {
          sb.append(obj.toString());
        }
        return sb.toString();
      }
    };
  }
 * </code>
 * 
 * @see org.springframework.cache.annotation.CachingConfigurer
 */
@Configuration
@ConditionalOnClass(CacheManager.class)
@ConditionalOnBean(CacheAspectSupport.class)
@AutoConfigureAfter({ CacheAutoConfiguration.class })
@ConditionalOnProperty(prefix = "spring.cache", name = "auto", havingValue = "true", matchIfMissing = true)
public class CacheAutoConfigurationAfter implements CachingConfigurer {
  private final CacheManager cacheManager;

  @Autowired
  public CacheAutoConfigurationAfter(CacheManager cacheManager) {
    this.cacheManager = cacheManager;
  }

  @Bean
  @Override
  public KeyGenerator keyGenerator() {
    return new KeyGeneratorImpl();
  }

  @Bean
  @Override
  public CacheResolver cacheResolver() {
    return new SimpleCacheResolver(this.cacheManager);
  }

  @Bean
  @Override
  public CacheErrorHandler errorHandler() {
    return new SimpleCacheErrorHandler();
  }

  @Override
  public CacheManager cacheManager() {
    return this.cacheManager;
  }
}
