package org.springframework.boot.actuate.health;

import java.io.File;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.boot.actuate.health.AbstractHealthIndicator;
import org.springframework.boot.actuate.health.Health.Builder;
import org.springframework.core.io.Resource;
import org.springframework.util.Assert;

/**
 * @see org.springframework.boot.actuate.health.DiskSpaceHealthIndicator
 */
public class MountHealthIndicator extends AbstractHealthIndicator {
  protected final Log logger = LogFactory.getLog(getClass());

  private final MountHealthIndicatorProperties properties;

  public MountHealthIndicator(MountHealthIndicatorProperties properties) {
    this.properties = properties;
  }

  /**
   * @see java.nio.file.Files#getFileStore(java.nio.file.Path)
   * @see java.nio.file.Paths#get(String, String...)
   * @see java.nio.file.FileStore#getUnallocatedSpace()
   */
  @Override
  protected void doHealthCheck(Builder builder) throws Exception {
    Resource resource = properties.getResource();
    Assert.notNull(resource, "'resource' must not be null ('management.health.mount.resource=file:/data'");
    Assert.isTrue(resource.exists(), "'resource' not exist from: " + resource);

    File file = resource.getFile().getCanonicalFile();
    builder.withDetail("file", file);

    Map<String, Object> access = new LinkedHashMap<String, Object>();
    access.put("access", properties.getAccess());
    access.put("execute", properties.isExecute());
    access.put("read", properties.isRead());
    access.put("write", properties.isWrite());
    builder.withDetail("access", access);

    Map<String, Object> type = new LinkedHashMap<String, Object>();
    type.put("type", properties.getType());
    type.put("file", properties.isFile());
    type.put("directory", properties.isDirectory());
    type.put("absolute", properties.isAbsolute());
    type.put("hidden", properties.isHidden());
    builder.withDetail("type", type);

    if (properties.isFile()) {
      Assert.isTrue(file.isFile(), "file not isFile");
    }
    if (properties.isDirectory()) {
      Assert.isTrue(file.isDirectory(), "file not isDirectory");
    }
    if (properties.isAbsolute()) {
      Assert.isTrue(file.isAbsolute(), "file not isAbsolute");
    }
    if (properties.isHidden()) {
      Assert.isTrue(file.isHidden(), "file not isHidden");
    }

    if (properties.isExecute()) {
      Assert.isTrue(file.canExecute(), "file not canExecute");
    }
    if (properties.isWrite()) {
      Assert.isTrue(file.canWrite(), "file not canWrite");
      if (properties.isDirectory()) {
        Assert.isTrue(File.createTempFile(UUID.randomUUID().toString() + '-', null, file).delete(), "file delete fail from: " + file);
      }
    }
    if (properties.isRead()) {
      Assert.isTrue(file.canRead(), "file not canRead");
    }
    builder.up();
  }
}