package kamino.sanguijuela

import java.net.InetAddress
import java.nio.ByteBuffer
import java.util

import org.asynchttpclient._
import kamino.cucaracha.model.ingestion.{MetricType, SnapshotBatch, Entity => CucarachaEntity, Metric => CucarachaMetric}

import scala.concurrent.Future
import akka.actor._
import akka.pattern.pipe
import kamon.Kamon
import kamon.metric.instrument.{Counter, Histogram, InstrumentSnapshot}
import kamon.metric.{Entity, EntitySnapshot, MetricKey, TickMetricSnapshotBuffer}
import kamon.metric.SubscriptionsDispatcher.TickMetricSnapshot
import org.HdrHistogram

import scala.collection.JavaConverters._
import java.util.{HashMap => JMap}

import akka.event.Logging

import scala.concurrent.duration._


object Sanguijuela extends ExtensionId[SanguijuelaExtension] with ExtensionIdProvider {
  override def lookup(): ExtensionId[_ <: Extension] = Sanguijuela
  override def createExtension(system: ExtendedActorSystem): SanguijuelaExtension = new SanguijuelaExtension(system)
}

class SanguijuelaExtension(system: ExtendedActorSystem) extends Kamon.Extension {
  val log = Logging(system, classOf[SanguijuelaExtension])
  val config = system.settings.config.getConfig("kamino")
  val flushInterval = 60 seconds
  val tickInterval = Kamon.metrics.settings.tickInterval
  val subscriber = system.actorOf(Props(
    new Subscriber(
      apiKey = config.getString("api-key"),
      applicationName = config.getString("application-name"),
      instance = InetAddress.getLocalHost.getHostName
    )),
    "kamino")

  val bufferedSubscriberProxy = if (tickInterval == flushInterval) {
    subscriber
  } else {
    system.actorOf(TickMetricSnapshotBuffer.props(flushInterval, subscriber), "kamino-metrics-buffer")
  }

  Kamon.metrics.subscribe("**", "**", bufferedSubscriberProxy, permanently = true)
  log.info(s"Starting the Kamon(Kamino) extension")
}

sealed trait IngestionResult
case object IngestionFailed extends IngestionResult
case object IngestionSucceeded extends IngestionResult

case class TryIngestion(data: Array[Byte])

class Subscriber(apiKey: String, applicationName: String, instance: String) extends Actor with Stash {
  implicit val ex = context.dispatcher
  val httpClient = new DefaultAsyncHttpClient()

  val config = context.system.settings.config.getConfig("kamino")
  val snapshotStorage = new OffHeapSnapshotStorage(config.getInt("snapshot-buffer-size"))
  val (host,port,path) = (config.getString("host"),config.getInt("port"),config.getString("route"))

  def receive = {
    case tick: TickMetricSnapshot => {
      snapshotStorage.put(encodeData(tick))
      snapshotStorage.getHead().foreach(data => {
        postIngestion(data)
        context.become(pushing)
      })
    }
  }

  def pushing: Actor.Receive = {
    case IngestionSucceeded => {
      snapshotStorage.popHead()
      snapshotStorage.getHead().map(data => {
        postIngestion(data)
      }).getOrElse({
        unstashAll()
        context.become(receive)
      })  //unstash all
    }
    case IngestionFailed => {
      unstashAll()
      context.become(receive)
    }
    case _ => stash()
  }


  def postIngestion(data: Array[Byte]) =
    pipe(postData(data)) to self

  private def postData(data: Array[Byte]): Future[IngestionResult] = {
    httpClient
      .prepareGet(s"https://$host:$port/$path")
      .setBody(data)
      .execute()
      .toFuture(context.dispatcher)
      .map(response => if(response.getStatusCode == 200) IngestionSucceeded else IngestionFailed)
      .recover { case _ => IngestionFailed }
  }


  override def postStop(): Unit = {
    httpClient.close()
  }

  private val _transferHistogram = new HdrHistogram.Histogram(3600000000000L, 2)
  private val _transferBuffer = ByteBuffer.allocate(_transferHistogram.getNeededByteBufferCapacity)

  def encodeData: TickMetricSnapshot => Array[Byte]  = tick => ModelSerializer.pack(toKaminoSnapshot(tick))

  def toKaminoSnapshot(tick: TickMetricSnapshot): SnapshotBatch = {
    def toKaminoMetric(metric: (MetricKey, InstrumentSnapshot)): CucarachaMetric = {
      val (metricKey, instrumentSnapshot) = metric

      // TODO: Modify Kamon so that we can have proper information about the keys.
      instrumentSnapshot match {
        case histogram: Histogram.Snapshot =>
          _transferHistogram.reset()
          _transferBuffer.clear()
          var measurementsCount = 0L

          histogram.recordsIterator.foreach { record =>
            for(_ <- 1 to record.count.toInt) {
              _transferHistogram.recordValue(record.level)
            }
            measurementsCount += record.count.toLong
          }

          // WARNING
          //
          //  This is a hack! We should find a way to optimize this, but currently we will just
          //  go through the long process of transforming the snapshots until Kamon is updated to
          //  use the same internal representation of the counts array.


          val rawSnapshotSize = _transferHistogram.encodeIntoByteBuffer(_transferBuffer)
          val hdrMetadataOffset = 40 // 40 bytes of internal state that we don't need.
          _transferBuffer.clear()
          _transferBuffer.position(hdrMetadataOffset)
          _transferBuffer.limit(rawSnapshotSize)

          val snapshotData = Array.ofDim[Byte](_transferBuffer.remaining())
          _transferBuffer.get(snapshotData)

          new CucarachaMetric(metricKey.name, MetricType.valueOf("HISTOGRAM"), snapshotData,measurementsCount)

        case counter: Counter.Snapshot  =>
          val buffer = ByteBuffer.wrap(Array.ofDim[Byte](8))
          buffer.putLong(counter.count)
          new CucarachaMetric(metricKey.name, MetricType.valueOf("COUNTER"), buffer.array(),counter.count)
      }
    }

    def toKaminoEntity(entity: (Entity, EntitySnapshot)): CucarachaEntity = {
      val (entityIdentity, snapshot) = entity
      val tags = new JMap[String,String](entityIdentity.tags.asJava)
      val metrics = new util.ArrayList[CucarachaMetric](snapshot.metrics.map(toKaminoMetric).toSeq.asJava)
      new CucarachaEntity(entityIdentity.category, entityIdentity.name, tags, metrics)
    }

    val entities = new util.ArrayList[CucarachaEntity](tick.metrics.map(toKaminoEntity).toSeq.asJava)
    new SnapshotBatch(apiKey, applicationName, instance, tick.from.millis, tick.to.millis, entities)
  }
}
