package io.ionic.liveupdates

/**
 * The LiveUpdate class represents an application registered with Live Updates in [Appflow](https://ionic.io/appflow).
 *
 * @property appId the ID of the app registered with Appflow
 * @property channelName the name of the channel used for updates
 * @property usesSecureLiveUpdates whether the app uses secured Live Updates
 * @property updateStrategy the strategy to use for downloading app updates (default DIFFERENTIAL)
 */
class LiveUpdate(val appId: String, val channelName: String = "production", val usesSecureLiveUpdates: Boolean = false, val updateStrategy: Strategy = Strategy.DIFFERENTIAL) {
    /**
     * The state of an available update.
     */
    var availableUpdate: AvailableUpdateState? = null
        internal set

    /**
     * The state of the app during the sync process.
     */
    var appState: AppState = AppState.UNCHECKED
        internal set

    /**
     * An asset path in the app APK that can be used as the source files for differential updates. The web assets should
     * contain a Live Update Manifest json.
     */
    var assetPath: String? = null

    constructor(appId: String, channelName: String = "production", usesSecureLiveUpdates: Boolean = false) : this(appId, channelName, usesSecureLiveUpdates, Strategy.DIFFERENTIAL)
    constructor(appId: String, channelName: String = "production") : this(appId, channelName, false)
    constructor(appId: String) : this(appId, "production", false)
}

/**
 * A state representing if there is an available update for an app.
 */
enum class AvailableUpdateState {
    /**
     * An update is available but not downloaded.
     */
    AVAILABLE,

    /**
     * An update is pending.
     */
    PENDING,

    /**
     * An app is updated and/or ready to use.
     */
    READY
}

/**
 * A state representing a step in the Live Update process.
 */
enum class AppState {
    /**
     * App has not been checked for update.
     */
    UNCHECKED,

    /**
     * Update check is in progress.
     */
    CHECKING,

    /**
     * App has been checked for update.
     */
    CHECKED,

    /**
     * An update is available but not downloaded.
     */
    AVAILABLE,

    /**
     * An update download is in process.
     */
    DOWNLOADING,

    /**
     * App update has been downloaded.
     */
    DOWNLOADED,

    /**
     * An update is copying duplicate files from a previous update (differential live updates only).
     */
    COPYING,

    /**
     * The update has finished copying duplicate files from a previous update (differential live updates only).
     */
    COPIED,

    /**
     * A downloaded update is in the process of being unpacked.
     */
    UNPACKING,

    /**
     * Update has been unpacked.
     */
    UNPACKED,

    /**
     * The app is being verified (secure Live Updates).
     */
    VERIFYING,

    /**
     * The app has been verified (secure Live Updates).
     */
    VERIFIED,

    /**
     * A downloaded unpacked update is being applied.
     */
    UPDATING,

    /**
     * The app is fully updated.
     */
    UPDATED,

    /**
     * The update process is canceled.
     */
    CANCELED,

    /**
     * The update process failed.
     */
    FAILED
}