package io.ionic.liveupdates.urltokenvalidation

import android.util.Base64
import io.jsonwebtoken.SignatureAlgorithm
import io.jsonwebtoken.impl.crypto.DefaultJwtSignatureValidator
import io.jsonwebtoken.io.Decoders
import kotlinx.serialization.Serializable
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.Json
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo
import org.bouncycastle.openssl.PEMParser
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter
import java.io.StringReader
import java.security.interfaces.RSAPublicKey

private val CUSTOM_URL_PUBLIC_KEY = """
    -----BEGIN PUBLIC KEY-----
    MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtGiiRnRuuUN4kRxyOFUi
    B+6Hm1iCbSqfIDtL7vAzTEIehVfMI2KVDb8pLmCG0rOX7vZ8TPkifZ4U9y7wAaj7
    9zm6IlBaELnK0uBW3B0kKT3ZXiFnN8CvesoHqljZ6ff+DBM8UYNvEmE5wSFryQJA
    iK3SgMJrGpEp3RM8yD0XDmbqNWhaQCN/BO3BREtAD/t9TmhZrarnC87fvQAqGcTE
    gKu5W9VRN9fiuc33rKN39iY+uk04Nzdqbg3Qchz+NsLDy4fkmCoQuF1zFL9B09gI
    73jUrQNnuyDye1IIB3Jz/EQhP35gii/nL75RGvSwMsv4egDA2NTAQpGFXL2zVJCk
    kwIDAQAB
    -----END PUBLIC KEY----- 
    """.trimIndent()

class PublicKeyException(message: String? = null, cause: Throwable? = null) : Exception(message, cause) {
    constructor(cause: Throwable) : this(null, cause)
}

class InvalidSignatureException(message: String? = null, cause: Throwable? = null) : Exception(message, cause) {
    constructor(cause: Throwable) : this(null, cause)
}

@Serializable
internal data class CustomUrlPayload(val android: Android, val url: String) {
    val packageNames: List<String>
        get() = android.packageNames
    @Serializable
    data class Android(val packageNames: List<String>)

    companion object {
        private val json = Json { ignoreUnknownKeys = true }

        fun fromToken(token: String): CustomUrlPayload {
            val validated = validateSignature(token)
            return json.decodeFromString(validated)
        }
    }
}

private fun validateSignature(token: String): String {
    val publicKey: RSAPublicKey
    try {
        val parser = PEMParser(StringReader(CUSTOM_URL_PUBLIC_KEY))
        val converter = JcaPEMKeyConverter()
        val publicKeyInfo = parser.readObject() as SubjectPublicKeyInfo
        publicKey = converter.getPublicKey(publicKeyInfo) as RSAPublicKey
    } catch (e: Exception) {
        throw PublicKeyException(e)
    }

    val splitToken = token.split(".")

    if (splitToken.size != 3) {
        throw PublicKeyException("JWT must be three Base64-URL encoded strings separated by periods.")
    }

    val (header, payload, signature) = splitToken

    val validator = DefaultJwtSignatureValidator(SignatureAlgorithm.RS256, publicKey, Decoders.BASE64URL)
    val result = validator.isValid("$header.$payload", signature)
    if (result) {
        return String(Base64.decode(payload, Base64.URL_SAFE))
    } else {
        throw InvalidSignatureException()
    }
}
