package io.ionic.liveupdates.network

import android.content.Context
import io.ionic.liveupdates.LiveUpdateManager
import io.ionic.liveupdates.Logger
import io.ionic.liveupdates.data.model.Manifest
import io.ionic.liveupdates.data.model.network.request.CheckRequest
import io.ionic.liveupdates.data.model.network.response.*
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import okhttp3.*
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.RequestBody.Companion.toRequestBody
import okio.IOException
import java.io.File
import java.io.FileOutputStream

internal object Client {
    private const val API_PROD = "https://api.ionicjs.com"
    private const val API_STAGE = "https://api-staging.ionicjs.com"

    internal var customUrl: String? = null

    private val client = OkHttpClient()

    private val json = Json { explicitNulls = false; encodeDefaults = true; isLenient = true }

    private fun getEndpointCheck(appId: String): String {
        return "${customUrl ?: API_PROD}/apps/${appId}/channels/check-device"
    }

    @Deprecated("Download URLs are provided by the API. Should no longer rely on this URL structure.")
    private fun getEndpointDownload(appId: String, snapshotID: String): String {
        return "${customUrl ?: API_PROD}/apps/${appId}/snapshots/${snapshotID}/download"
    }

    fun checkForUpdate(checkRequest: CheckRequest): CheckResponse? {
        val requestBody = json.encodeToString(checkRequest)
            .toRequestBody("application/json; charset=utf-8".toMediaType())
        val request = Request.Builder()
            .url(getEndpointCheck(checkRequest.app_id))
            .post(requestBody)
            .build()

        try {
            client.newCall(request).execute().use { response ->
                val responseCode = response.code
                val responseBody = response.body?.string()
                response.close()

                if (!response.isSuccessful) {
                    if (responseBody != null) {
                        Logger.error("App ID - ${checkRequest.app_id}: Check update request failed with response code $responseCode")
                        Logger.error("App ID - ${checkRequest.app_id}: Error Response Body: $responseBody")

                        try {
                            val errorResponse = json.decodeFromString<ErrorResponse>(responseBody)
                            return CheckResponse(null, errorResponse)
                        } catch (e: Exception) {
                            return CheckResponse(
                                null,
                                ErrorResponse(
                                    Error(
                                        null,
                                        "Check request failed for $responseBody with response code $responseCode",
                                        "Server error",
                                        null,
                                        null
                                    ), null
                                )
                            )
                        }
                    } else {
                        Logger.error("App ID - ${checkRequest.app_id}: Check update request failed with response code $responseCode")
                        return CheckResponse(
                            null,
                            ErrorResponse(
                                Error(
                                    null,
                                    "Check request failed with response code $responseCode",
                                    "Server error",
                                    null,
                                    null
                                ), null
                            )
                        )
                    }
                }

                if (responseBody != null) {
                    try {
                        val successResponse = json.decodeFromString<SuccessResponse>(responseBody)
                        return CheckResponse(successResponse, null)
                    } catch (e: Exception) {
                        Logger.error("App ID - ${checkRequest.app_id}: Check update request succeeded but the response could not be parsed.")
                        Logger.error("App ID - ${checkRequest.app_id}: Response Body: $responseBody")
                        return CheckResponse(
                            null,
                            ErrorResponse(
                                Error(
                                    null,
                                    "Check request failed for $responseBody with response code $responseCode",
                                    "Server error",
                                    null,
                                    null
                                ), null
                            )
                        )
                    }
                }

                Logger.error("App ID - ${checkRequest.app_id}: Check update request succeeded but the response was empty.")
                return CheckResponse(
                    null,
                    ErrorResponse(
                        Error(
                            null,
                            "Check request failed with server code $responseCode. Empty body.",
                            "Server error",
                            null,
                            null
                        ), null
                    )
                )
            }
        } catch (e: Exception) {
            return CheckResponse(
                null,
                ErrorResponse(Error(null, e.toString(), "Exception", null, null), null)
            )
        }
    }

    @Deprecated(
        "Newer download method uses URL provided by the API.",
        ReplaceWith("downloadUpdate(context, appId, snapshotID, downloadURL)"),
        DeprecationLevel.WARNING
    )
    fun downloadUpdate(context: Context, appId: String, snapshotID: String): DownloadResponse {
        return downloadUpdate(context, appId, snapshotID, getEndpointDownload(appId, snapshotID))
    }

    fun downloadUpdate(
        context: Context,
        appId: String,
        snapshotId: String,
        downloadURL: String
    ): DownloadResponse {
        val appDir = File(LiveUpdateManager.getLiveUpdatesDirectory(context), appId)

        val request = Request.Builder()
            .url(downloadURL)
            .build()

        try {
            client.newCall(request).execute().use { response ->
                if (!response.isSuccessful) {
                    val responseCode = response.code
                    val responseBody = response.body?.string()
                    response.close()

                    Logger.error("App ID - $appId, Snapshot ID - $snapshotId: Download update request failed with response code $responseCode.")

                    if (!responseBody.isNullOrEmpty()) {
                        Logger.error("App ID - $appId, Snapshot ID - $snapshotId: Error Response Body: $responseBody")

                        try {
                            val errorResponse = json.decodeFromString<ErrorResponse>(responseBody)
                            return DownloadResponse(errorResponse, null)
                        } catch (e: Exception) {
                            return DownloadResponse(
                                ErrorResponse(
                                    Error(
                                        null,
                                        "Download request failed for $appId snapshot $snapshotId with response code $responseCode. Body: $responseBody",
                                        "Server error",
                                        null,
                                        null
                                    ), null
                                ), null
                            )
                        }
                    }

                    Logger.error("App ID - $appId, Snapshot ID - $snapshotId: No Error Response Body.")
                    return DownloadResponse(
                        ErrorResponse(
                            Error(
                                null,
                                "Download response not successful but no error body provided. Response code $responseCode",
                                "Server Error",
                                null,
                                null
                            ), null
                        ), null
                    )
                }

                try {
                    val fileData = response.body?.byteStream()
                    if (fileData != null) {
                        val fileOutputStream = FileOutputStream(File(appDir, "${snapshotId}.zip"))
                        val buffer = ByteArray(8192)
                        var bytesRead: Int

                        while (fileData.read(buffer).also { bytesRead = it } != -1) {
                            fileOutputStream.write(buffer, 0, bytesRead)
                        }

                        fileOutputStream.close()
                        fileData.close()
                        Logger.debug("App ID - $appId, Snapshot ID - $snapshotId: Snapshot zip downloaded.")
                    }
                } catch (exception: IOException) {
                    Logger.error("App ID - $appId, Snapshot ID - $snapshotId: Saving download failed.")
                    Logger.error(exception.stackTraceToString())
                    return DownloadResponse(
                        ErrorResponse(
                            Error(
                                null,
                                exception.toString(),
                                "Exception",
                                null,
                                null
                            ), null
                        ), null
                    )
                }

                return DownloadResponse(null, File(appDir, "${snapshotId}.zip"))
            }
        } catch (exception: Exception) {
            return DownloadResponse(
                ErrorResponse(
                    Error(
                        null,
                        exception.toString(),
                        "Exception",
                        null,
                        null
                    ), null
                ), null
            )
        }
    }

    fun getManifest(appId: String, snapshotId: String, downloadURL: String): ManifestResponse {
        val request = Request.Builder()
            .url(downloadURL)
            .build()

        try {
            client.newCall(request).execute().use { response ->
                if (!response.isSuccessful) {
                    val responseCode = response.code
                    val responseBody = response.body?.string()
                    response.close()

                    Logger.error("App ID - $appId, Snapshot ID - $snapshotId: Manifest download request failed with response code $responseCode.")

                    if (!responseBody.isNullOrEmpty()) {
                        Logger.error("App ID - $appId, Snapshot ID - $snapshotId: Error Response Body: $responseBody")

                        try {
                            val errorResponse = json.decodeFromString<ErrorResponse>(responseBody)
                            return ManifestResponse(errorResponse, null, null, null, null)
                        } catch (e: Exception) {
                            return ManifestResponse(
                                ErrorResponse(
                                    Error(
                                        null,
                                        "Manifest download request failed for app $appId snapshot $snapshotId with response code $responseCode. Body: $responseBody",
                                        "Server error",
                                        null,
                                        null
                                    ), null
                                ), null, null, null, null
                            )
                        }
                    }

                    Logger.error("App ID - $appId, Snapshot ID - $snapshotId: No Error Response Body.")
                    return ManifestResponse(
                        ErrorResponse(
                            Error(
                                null,
                                "Manifest response not successful but no error body provided. Response code $responseCode",
                                "Server Error",
                                null,
                                null
                            ), null
                        ), null, null, null, null
                    )
                } else {
                    try {
                        val responseBody = response.body?.string()
                        response.close()

                        if (!responseBody.isNullOrEmpty()) {
                            val files = json.decodeFromString<Manifest>(responseBody)
                            val responseQuery = response.request.url.query

                            var responseURL = response.request.url.toUrl().toString()
                            responseURL = responseURL.substringBefore("?")
                            responseURL = responseURL.substringBeforeLast("/")

                            var manifestHref = response.request.url.toUrl().toString()
                            manifestHref = manifestHref.substringBefore("?")
                            manifestHref = manifestHref.substringAfterLast("/")

                            return ManifestResponse(
                                null,
                                files.decodedPayload,
                                manifestHref,
                                responseURL,
                                responseQuery
                            )
                        }
                    } catch (exception: IOException) {
                        Logger.error("App ID - $appId, Snapshot ID - $snapshotId: Saving manifest failed.")
                        Logger.error(exception.stackTraceToString())
                        return ManifestResponse(
                            ErrorResponse(
                                Error(
                                    null,
                                    exception.toString(),
                                    "Exception",
                                    null,
                                    null
                                ), null
                            ), null, null, null, null
                        )
                    }
                }
            }
        } catch (exception: Exception) {
            return ManifestResponse(
                ErrorResponse(
                    Error(
                        null,
                        exception.toString(),
                        "Exception",
                        null,
                        null
                    ), null
                ), null, null, null, null
            )
        }

        return ManifestResponse(null, null, null, null, null)
    }

    fun downloadFile(
        appId: String,
        snapshotId: String,
        path: String,
        downloadURL: String,
        query: String?,
        destination: File
    ): DownloadResponse {
        var requestQuery = ""
        if (!query.isNullOrEmpty()) {
            requestQuery = "?${query}"
        }
        val request = Request.Builder()
            .url("${downloadURL}${requestQuery}")
            .build()

        Logger.debug("App ID - $appId, Snapshot ID - $snapshotId: File $path download starting.")

        try {
            client.newCall(request).execute().use { response ->
                if (!response.isSuccessful) {
                    val responseCode = response.code
                    val responseBody = response.body?.string()
                    response.close()

                    if (!responseBody.isNullOrEmpty()) {
                        Logger.error("App ID - $appId, Snapshot ID - $snapshotId: Download file $path request failed with response code $responseCode.")
                        Logger.error("App ID - $appId, Snapshot ID - $snapshotId: Error Response Body: $responseBody")

                        try {
                            val errorResponse = json.decodeFromString<ErrorResponse>(responseBody)
                            return DownloadResponse(errorResponse, null)
                        } catch (e: Exception) {
                            return DownloadResponse(
                                ErrorResponse(
                                    Error(
                                        null,
                                        "Download request failed for $appId snapshot $snapshotId with response code $responseCode. Body: $responseBody",
                                        "Server error",
                                        null,
                                        null
                                    ), null
                                ), null
                            )
                        }
                    }

                    return DownloadResponse(
                        ErrorResponse(
                            Error(
                                null,
                                "Download response not successful but no error body provided.",
                                "Server Error",
                                null,
                                null
                            ), null
                        ), null
                    )
                }

                try {
                    val fileData = response.body?.byteStream()
                    if (fileData != null) {
                        val file = File(destination, path)
                        if (file.exists()) {
                            file.delete()
                        } else {
                            val makeDirsResult = File(destination, path).parent?.let { File(it).mkdirs() }
                        }

                        val fileOutputStream = FileOutputStream(File(destination, path))
                        val buffer = ByteArray(8192)
                        var bytesRead: Int

                        while (fileData.read(buffer).also { bytesRead = it } != -1) {
                            fileOutputStream.write(buffer, 0, bytesRead)
                        }

                        fileOutputStream.close()
                        fileData.close()
                        Logger.debug("App ID - $appId, Snapshot ID - $snapshotId: File $path downloaded.")
                    }
                    response.close()
                } catch (exception: IOException) {
                    Logger.error("App ID - $appId, Snapshot ID - $snapshotId: Saving file $path failed.")
                    Logger.error(exception.stackTraceToString())
                    return DownloadResponse(
                        ErrorResponse(
                            Error(
                                null,
                                exception.toString(),
                                "Exception",
                                null,
                                null
                            ), null
                        ), null
                    )
                }

                return DownloadResponse(null, File(destination, path))
            }
        } catch (exception: Exception) {
            return DownloadResponse(
                ErrorResponse(
                    Error(
                        null,
                        exception.toString(),
                        "Exception",
                        null,
                        null
                    ), null
                ), null
            )
        }
    }
}