package io.ionic.liveupdates.network

import android.content.Context
import io.ionic.liveupdates.LiveUpdateManager
import io.ionic.liveupdates.Logger
import io.ionic.liveupdates.data.model.Manifest
import io.ionic.liveupdates.data.model.network.request.CheckRequest
import io.ionic.liveupdates.data.model.network.response.*
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import okhttp3.*
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.RequestBody.Companion.toRequestBody
import okio.IOException
import java.io.File
import java.io.FileOutputStream

internal object Client {
    private const val API_PROD = "https://api.ionicjs.com"
    private const val API_STAGE = "https://api-staging.ionicjs.com"

    private val client = OkHttpClient()

    private val json = Json { explicitNulls = false; encodeDefaults = true; isLenient = true }

    private fun getEndpointCheck(appId: String): String {
        return "${API_PROD}/apps/${appId}/channels/check-device"
    }

    @Deprecated("Download URLs are provided by the API. Should no longer rely on this URL structure.")
    private fun getEndpointDownload(appId: String, snapshotID: String): String {
        return "${API_PROD}/apps/${appId}/snapshots/${snapshotID}/download"
    }

    fun checkForUpdate(checkRequest: CheckRequest): CheckResponse? {
        val requestBody = json.encodeToString(checkRequest)
            .toRequestBody("application/json; charset=utf-8".toMediaType())
        val request = Request.Builder()
            .url(getEndpointCheck(checkRequest.app_id))
            .post(requestBody)
            .build()

        try {
            client.newCall(request).execute().use { response ->
                val bodyObj = response.body
                if (!response.isSuccessful) {
                    if (bodyObj != null) {
                        val bodyString = bodyObj.string()
                        try {
                            val errorResponse = json.decodeFromString<ErrorResponse>(bodyString)
                            Logger.error("Check update request failed for " + checkRequest.app_id)
                            Logger.error(bodyString)
                            response.close()
                            return CheckResponse(null, errorResponse)
                        } catch (e: Exception) {
                            Logger.error("Check update request failed for " + checkRequest.app_id + " for " + bodyString + " with request code " + response.code)
                            response.close()
                            return CheckResponse(
                                null,
                                ErrorResponse(
                                    Error(
                                        null,
                                        "Check request failed for " + bodyString + " with request code " + response.code,
                                        "Server error",
                                        null,
                                        null
                                    ), null
                                )
                            )
                        }
                    } else {
                        Logger.error("Check update request failed for " + checkRequest.app_id + " with request code " + response.code)
                        response.close()
                        return CheckResponse(
                            null,
                            ErrorResponse(
                                Error(
                                    null,
                                    "Check request failed with server code " + response.code,
                                    "Server error",
                                    null,
                                    null
                                ), null
                            )
                        )
                    }
                }

                if (bodyObj != null) {
                    val bodyString = bodyObj.string()
                    val successResponse = json.decodeFromString<SuccessResponse>(bodyString)
                    response.close()
                    return CheckResponse(successResponse, null)
                }

                response.close()
                return CheckResponse(
                    null,
                    ErrorResponse(
                        Error(
                            null,
                            "Check request failed with server code " + response.code,
                            "Server error",
                            null,
                            null
                        ), null
                    )
                )
            }
        } catch (e: Exception) {
            return CheckResponse(
                null,
                ErrorResponse(Error(null, e.toString(), "Exception", null, null), null)
            )
        }
    }

    @Deprecated(
        "Newer download method uses URL provided by the API.",
        ReplaceWith("downloadUpdate(context, appId, snapshotID, downloadURL)"),
        DeprecationLevel.WARNING
    )
    fun downloadUpdate(context: Context, appId: String, snapshotID: String): DownloadResponse {
        return downloadUpdate(context, appId, snapshotID, getEndpointDownload(appId, snapshotID))
    }

    fun downloadUpdate(
        context: Context,
        appId: String,
        snapshotId: String,
        downloadURL: String
    ): DownloadResponse {
        val appDir = File(LiveUpdateManager.getLiveUpdatesDirectory(context), appId)

        val request = Request.Builder()
            .url(downloadURL)
            .build()

        try {
            client.newCall(request).execute().use { response ->
                if (!response.isSuccessful) {
                    val responseBody = response.body
                    if (responseBody != null) {
                        val bodyString = responseBody.string()
                        val errorResponse = json.decodeFromString<ErrorResponse>(bodyString)
                        Logger.error("Download update request failed for app $appId snapshot $snapshotId")
                        Logger.error(bodyString)
                        response.close()
                        return DownloadResponse(errorResponse, null)
                    }

                    response.close()
                    return DownloadResponse(
                        ErrorResponse(
                            Error(
                                null,
                                "Download response not successful but no error body provided.",
                                "Server Error",
                                null,
                                null
                            ), null
                        ), null
                    )
                }

                try {
                    val fileData = response.body?.byteStream()
                    if (fileData != null) {
                        val fileOutputStream = FileOutputStream(File(appDir, "${snapshotId}.zip"))
                        fileOutputStream.write(fileData.readBytes())
                        fileOutputStream.close()
                        fileData.close()
                        Logger.debug("Snapshot ${snapshotId}.zip downloaded for app $appId")
                    }
                } catch (exception: IOException) {
                    Logger.error("Saving download failed for app $appId snapshot $snapshotId")
                    Logger.error(exception.stackTraceToString())
                    response.close()
                    return DownloadResponse(
                        ErrorResponse(
                            Error(
                                null,
                                exception.toString(),
                                "Exception",
                                null,
                                null
                            ), null
                        ), null
                    )
                }

                response.close()
                return DownloadResponse(null, File(appDir, "${snapshotId}.zip"))
            }
        } catch (exception: Exception) {
            return DownloadResponse(
                ErrorResponse(
                    Error(
                        null,
                        exception.toString(),
                        "Exception",
                        null,
                        null
                    ), null
                ), null
            )
        }
    }

    fun getManifest(appId: String, snapshotId: String, downloadURL: String): ManifestResponse {
        val request = Request.Builder()
            .url(downloadURL)
            .build()

        try {
            client.newCall(request).execute().use { response ->
                if (!response.isSuccessful) {
                    val responseBody = response.body
                    if (responseBody != null) {
                        val bodyString = responseBody.string()
                        val errorResponse = json.decodeFromString<ErrorResponse>(bodyString)
                        Logger.error("Manifest download request failed for app $appId snapshot $snapshotId")
                        Logger.error(bodyString)
                        response.close()
                        return ManifestResponse(errorResponse, null, null, null, null)
                    }

                    response.close()
                    return ManifestResponse(
                        ErrorResponse(
                            Error(
                                null,
                                "Manifest response not successful but no error body provided.",
                                "Server Error",
                                null,
                                null
                            ), null
                        ), null, null, null, null
                    )
                } else {
                    try {
                        if (response.body != null) {
                            val bodyString = response.body!!.string()
                            val files = json.decodeFromString<Manifest>(bodyString)
                            val responseQuery = response.request.url.query

                            var responseURL = response.request.url.toUrl().toString()
                            responseURL = responseURL.substringBefore("?")
                            responseURL = responseURL.substringBeforeLast("/")

                            var manifestHref = response.request.url.toUrl().toString()
                            manifestHref = manifestHref.substringBefore("?")
                            manifestHref = manifestHref.substringAfterLast("/")

                            response.close()
                            return ManifestResponse(
                                null,
                                files.decodedPayload,
                                manifestHref,
                                responseURL,
                                responseQuery
                            )
                        }
                    } catch (exception: IOException) {
                        Logger.error("Saving manifest failed for app $appId snapshot $snapshotId")
                        Logger.error(exception.stackTraceToString())
                        response.close()
                        return ManifestResponse(
                            ErrorResponse(
                                Error(
                                    null,
                                    exception.toString(),
                                    "Exception",
                                    null,
                                    null
                                ), null
                            ), null, null, null, null
                        )
                    }
                }
            }
        } catch (exception: Exception) {
            return ManifestResponse(
                ErrorResponse(
                    Error(
                        null,
                        exception.toString(),
                        "Exception",
                        null,
                        null
                    ), null
                ), null, null, null, null
            )
        }

        return ManifestResponse(null, null, null, null, null)
    }

    fun downloadFile(
        appId: String,
        snapshotId: String,
        path: String,
        downloadURL: String,
        query: String?,
        destination: File
    ): DownloadResponse {
        var requestQuery = ""
        if (!query.isNullOrEmpty()) {
            requestQuery = "?${query}"
        }
        val request = Request.Builder()
            .url("${downloadURL}${requestQuery}")
            .build()

        Logger.debug("File $path download starting for app $appId")

        try {
            client.newCall(request).execute().use { response ->
                if (!response.isSuccessful) {
                    val responseBody = response.body
                    if (responseBody != null) {
                        val bodyString = responseBody.string()
                        val errorResponse = json.decodeFromString<ErrorResponse>(bodyString)
                        Logger.error("Download file $path request failed for app $appId snapshot $snapshotId")
                        Logger.error(bodyString)
                        response.close()
                        return DownloadResponse(errorResponse, null)
                    }

                    response.close()
                    return DownloadResponse(
                        ErrorResponse(
                            Error(
                                null,
                                "Download response not successful but no error body provided.",
                                "Server Error",
                                null,
                                null
                            ), null
                        ), null
                    )
                }

                try {
                    val fileData = response.body?.byteStream()
                    if (fileData != null) {
                        val file = File(destination, path)
                        if (file.exists()) {
                            file.delete()
                        } else {
                            val makeDirsResult = File(destination, path).parent?.let { File(it).mkdirs() }
                        }

                        val fileOutputStream = FileOutputStream(File(destination, path))
                        fileOutputStream.write(fileData.readBytes())
                        fileOutputStream.close()
                        fileData.close()
                        Logger.debug("File $path downloaded for app $appId")
                    }
                } catch (exception: IOException) {
                    Logger.error("Saving file $path failed for app $appId snapshot $snapshotId")
                    Logger.error(exception.stackTraceToString())
                    response.close()
                    return DownloadResponse(
                        ErrorResponse(
                            Error(
                                null,
                                exception.toString(),
                                "Exception",
                                null,
                                null
                            ), null
                        ), null
                    )
                }

                response.close()
                return DownloadResponse(null, File(destination, path))
            }
        } catch (exception: Exception) {
            return DownloadResponse(
                ErrorResponse(
                    Error(
                        null,
                        exception.toString(),
                        "Exception",
                        null,
                        null
                    ), null
                ), null
            )
        }
    }
}