package io.ionic.liveupdates

import android.content.Context
import android.util.Base64
import io.ionic.liveupdates.data.DataManager
import io.ionic.liveupdates.data.model.*
import io.ionic.liveupdates.data.model.network.response.CheckResponse
import io.ionic.liveupdates.data.model.network.response.DownloadResponse
import io.ionic.liveupdates.data.model.network.response.ManifestResponse
import io.ionic.liveupdates.network.*
import kotlinx.coroutines.*
import kotlinx.coroutines.sync.Semaphore
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.decodeFromStream
import okio.HashingSink
import okio.blackholeSink
import okio.buffer
import okio.source
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo
import org.bouncycastle.openssl.PEMParser
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter
import org.bouncycastle.util.io.pem.PemReader
import java.io.*
import java.io.File.separator
import java.security.Signature
import java.security.interfaces.RSAPublicKey
import java.util.concurrent.ConcurrentHashMap
import kotlin.coroutines.coroutineContext

/**
 * The Live Update Manager is used to manage instances of web apps used with [Live Updates](https://ionic.io/appflow).
 * It follows a [Singleton Pattern](https://en.wikipedia.org/wiki/Singleton_pattern) to allow access from anywhere in the application.
 * Use the Live Update Manager to configure and run sync to update web apps, check the status of an active sync, and get the file
 * path of the latest web assets on the device.
 */
object LiveUpdateManager {

    /**
     * Parent directory used by live updates to store web app files.
     */
    private const val LIVE_UPDATES_DIR = "ionic_apps"

    /**
     * Manifest file used for Secure Live Updates.
     */
    private const val SECURE_LIVE_UPDATES_MANIFEST = "live-update-manifest.json"

    /**
     * Apps to maintain using Live Updates.
     */
    private val instances: MutableMap<String, LiveUpdate> = mutableMapOf()

    /**
     * The maximum number of updates per app instance to be stored locally on the device.
     */
    var maxVersions: Int = 3

    /**
     * The name of the file containing the public key used to validate Secure Live Updates.
     */
    var secureLiveUpdatePEM = "ionic_cloud_public.pem"

    /**
     * General coroutine scope.
     */
    private val liveUpdateScope = CoroutineScope(Dispatchers.Default)

    /**
     * Coroutine scope used for network and IO operations.
     */
    private val liveUpdateIOScope = CoroutineScope(Dispatchers.IO)

    /**
     * Coroutine scope used for UI operations.
     */
    private val mainScope = CoroutineScope(Dispatchers.Main)

    /**
     * Track any running sync jobs
     */
    private val syncJobs: ConcurrentHashMap<String, Job> = ConcurrentHashMap()

    /**
     * Semaphore used to limit the sync operation to once per app at a time
     */
    private val syncSemaphore = Semaphore(1)

    /**
     * Configure logging output for the Live Updates library. This is enabled by default.
     * If set to false, logging is disabled and should not print to console.
     */
    @JvmStatic
    fun loggingEnabled(enabled: Boolean) {
        Logger.isLoggingEnabled = enabled
    }

    /**
     * Initializes the file directory and shared preferences used to save update data.
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.initialize(context)
     * ```
     *
     * Example usage (Java):
     * ```java
     * LiveUpdateManager.initialize(context);
     * ```
     *
     * @param context an Android [Context] used to initialize
     */
    @JvmStatic
    fun initialize(context: Context) {
        setupMainDirectory(context)
        DataManager.initialize(context)
    }

    /**
     * Clears the live updates directory and saved update data.
     * Attempts to cancel any running sync jobs.
     *
     * Example usage (kotlin):
     * ```kotlin
     * // Reset all apps
     * LiveUpdateManager.reset(context)
     *
     * // Reset but retain downloaded app files
     * LiveUpdateManager.reset(context, true)
     * ```
     *
     * Example usage (Java):
     * ```java
     * // Reset all apps
     * LiveUpdateManager.reset(context);
     *
     * // Reset but retain downloaded app files
     * LiveUpdateManager.reset(context, true);
     * ```
     *
     * @param context an Android [Context] used to reset
     * @param retainCache retain downloaded app files and live update cache (false by default)
     */
    @JvmStatic
    @JvmOverloads
    fun reset(context: Context, retainCache: Boolean = false) {
        // Attempt to cancel any running sync jobs
        cancelSync()

        // Recreate main and app directories
        setupMainDirectory(context, reset = !retainCache)
        instances.map {
            setupAppDirectory(context, it.key)
            it.value.appState = AppState.UNCHECKED
        }

        // Recreate shared preferences
        DataManager.reset(context, retainCache)
        for (instance in instances) {
            storeAppData(context, instance.value)
        }

        Logger.debug("Live Updates data has been reset!")
    }

    /**
     * Adds an app to the LiveUpdateManager.
     *
     * Example usage (kotlin):
     * ```kotlin
     * val liveUpdateConfig = LiveUpdate("appId", "production")
     * LiveUpdateManager.addLiveUpdateInstance(context, liveUpdateConfig)
     * ```
     *
     * Example usage (Java):
     * ```java
     * LiveUpdate liveUpdateConfig = new LiveUpdate("appId", "production");
     * LiveUpdateManager.addLiveUpdateInstance(context, liveUpdateConfig);
     * ```
     *
     * @param context an Android [Context] used to store app data
     * @param liveUpdate an instance of an app [LiveUpdate] to register with Live Updates
     */
    @JvmStatic
    fun addLiveUpdateInstance(context: Context, liveUpdate: LiveUpdate) {
        instances[liveUpdate.appId] = liveUpdate
        storeAppData(context, liveUpdate)
        setupAppDirectory(context, liveUpdate.appId)
    }

    /**
     * Check when an app was last synced.
     *
     * Example usage (kotlin):
     * ```kotlin
     * val lastSync = LiveUpdateManager.getLastSync(context, "appId")
     * ```
     *
     * Example usage (Java):
     * ```java
     * Long lastSync = LiveUpdateManager.getLastSync(context, "appId");
     * ```
     * @param context an Android [Context] used to get data from shared preferences
     * @param appId the ID of the app registered with Live Updates
     * @return the timestamp of the last sync in milliseconds, or -1 if no prior sync
     */
    @JvmStatic
    fun getLastSync(context: Context, appId: String): Long {
        return DataManager.getLastSync(context, appId)
    }

    /**
     * Check the latest time any of the registered apps was synced.
     *
     * Example usage (kotlin):
     * ```kotlin
     * val lastSync = LiveUpdateManager.getLastSync(context)
     * ```
     *
     * Example usage (Java):
     * ```java
     * Long lastSync = LiveUpdateManager.getLastSync(context);
     * ```
     *
     * @param context an Android [Context] used to get data from shared preferences
     * @return the timestamp of the oldest synced registered app, or -1 if any app has never been synced
     */
    @JvmStatic
    fun getLastSync(context: Context): Long {
        var oldestSync = Long.MAX_VALUE
        instances.map { instance ->
            val app = DataManager.getApp(context, instance.value.appId)
            val appSyncTime = app?.lastSync
            if (appSyncTime != null && appSyncTime < oldestSync) {
                oldestSync = appSyncTime
            }
        }

        return if (oldestSync == Long.MAX_VALUE) {
            -1
        } else {
            oldestSync
        }
    }

    /**
     * Checks with live updates to see if any new builds are available. If they are,
     * attempt to download, unzip, and update the saved data to reference the latest build.
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.sync(context)
     *
     * // Sync with a callback
     * LiveUpdateManager.sync(context, callback = object : SyncCallback {
     *     override fun onAppComplete(syncResult: SyncResult) {
     *         Log.d("LiveUpdate","CALLBACK: Sync success for app ${syncResult.liveUpdate.appId}!")
     *     }
     *
     *     override fun onAppComplete(failResult: FailResult) {
     *         Log.e("LiveUpdate","CALLBACK: Sync failed at step ${failResult.failStep.name} for app ${failResult.liveUpdate.appId}!")
     *     }
     *
     *     override fun onSyncComplete() {
     *         Log.d("LiveUpdate","CALLBACK: Sync finished!")
     *     }
     * })
     * ```
     *
     * Example usage (Java):
     * ```java
     * Long lastSync = LiveUpdateManager.getLastSync(context);
     *
     * // Sync with a callback
     * LiveUpdateManager.sync(this, new SyncCallback() {
     *     @Override
     *     public void onAppComplete(@NonNull SyncResult syncResult) {
     *         Log.d("LiveUpdate","CALLBACK: Sync success for app " + syncResult.getLiveUpdate().getAppId());
     *     }
     *
     *     @Override
     *     public void onAppComplete(@NonNull FailResult failResult) {
     *         Log.d("LiveUpdate","CALLBACK: Sync failed at step " + failResult.getFailStep().name() + " for app " + failResult.getLiveUpdate().getAppId());
     *     }
     *
     *     @Override
     *     public void onSyncComplete() {
     *         Log.d("LiveUpdate","CALLBACK: Sync finished!");
     *     }
     * });
     * ```
     *
     * @param context an Android [Context] used during the app update operation
     * @param callback a [SyncCallback] to notify on each app sync complete and when sync is complete
     */
    @JvmStatic
    fun sync(context: Context, callback: SyncCallback? = null) {
        sync(context, arrayOf(), callback = callback)
    }

    /**
     * Checks a single app with live updates to see if any new builds are available. If they are,
     * attempt to download, unzip, and update the saved data to reference the latest build.
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.sync(context, "appId")
     *
     * // Sync with a callback
     * LiveUpdateManager.sync(context, "appId", callback = object : SyncCallback {
     *     override fun onAppComplete(syncResult: SyncResult) {
     *         Log.d("LiveUpdate","CALLBACK: Sync success for app ${syncResult.liveUpdate.appId}!")
     *     }
     *
     *     override fun onAppComplete(failResult: FailResult) {
     *         Log.e("LiveUpdate","CALLBACK: Sync failed at step ${failResult.failStep.name} for app ${failResult.liveUpdate.appId}!")
     *     }
     *
     *     override fun onSyncComplete() {
     *         Log.d("LiveUpdate","CALLBACK: Sync finished!")
     *     }
     * })
     * ```
     *
     * Example usage (Java):
     * ```java
     * Long lastSync = LiveUpdateManager.getLastSync(context, "appId");
     *
     * // Sync with a callback
     * LiveUpdateManager.sync(this, "appId", new SyncCallback() {
     *     @Override
     *     public void onAppComplete(@NonNull SyncResult syncResult) {
     *         Log.d("LiveUpdate","CALLBACK: Sync success for app " + syncResult.getLiveUpdate().getAppId());
     *     }
     *
     *     @Override
     *     public void onAppComplete(@NonNull FailResult failResult) {
     *         Log.d("LiveUpdate","CALLBACK: Sync failed at step " + failResult.getFailStep().name() + " for app " + failResult.getLiveUpdate().getAppId());
     *     }
     *
     *     @Override
     *     public void onSyncComplete() {
     *         Log.d("LiveUpdate","CALLBACK: Sync finished!");
     *     }
     * });
     * ```
     *
     * @param context an Android [Context] used during the app update operation
     * @param appId a registered appId to sync
     * @param callback a [SyncCallback] to notify on each app sync complete and when sync is complete
     */
    @JvmStatic
    @JvmOverloads
    fun sync(context: Context, appId: String, callback: SyncCallback? = null) {
        sync(context, arrayOf(appId), callback = callback)
    }

    /**
     * Checks all apps or a provided set of apps with live updates to see if any new builds are
     * available. If they are, attempt to download, unzip, and update the saved data to reference
     * the latest build.
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.sync(context, arrayOf("appId1", "appId2"))
     *
     * // Sync with a callback
     * LiveUpdateManager.sync(context, arrayOf("appId1", "appId2"), async = false, callback = object : SyncCallback {
     *     override fun onAppComplete(syncResult: SyncResult) {
     *         Log.d("LiveUpdate","CALLBACK: Sync success for app ${syncResult.liveUpdate.appId}!")
     *     }
     *
     *     override fun onAppComplete(failResult: FailResult) {
     *         Log.e("LiveUpdate","CALLBACK: Sync failed at step ${failResult.failStep.name} for app ${failResult.liveUpdate.appId}!")
     *     }
     *
     *     override fun onSyncComplete() {
     *         Log.d("LiveUpdate","CALLBACK: Sync finished!")
     *     }
     * })
     * ```
     *
     * Example usage (Java):
     * ```java
     * Long lastSync = LiveUpdateManager.getLastSync(context, new String[] {"appId1", "appId2"});
     *
     * // Sync with a callback
     * LiveUpdateManager.sync(this, new String[] {"appId1", "appId2"}, false, new SyncCallback() {
     *     @Override
     *     public void onAppComplete(@NonNull SyncResult syncResult) {
     *         Log.d("LiveUpdate","CALLBACK: Sync success for app " + syncResult.getLiveUpdate().getAppId());
     *     }
     *
     *     @Override
     *     public void onAppComplete(@NonNull FailResult failResult) {
     *         Log.d("LiveUpdate","CALLBACK: Sync failed at step " + failResult.getFailStep().name() + " for app " + failResult.getLiveUpdate().getAppId());
     *     }
     *
     *     @Override
     *     public void onSyncComplete() {
     *         Log.d("LiveUpdate","CALLBACK: Sync finished!");
     *     }
     * });
     * ```
     *
     * @param context an Android [Context] used during the app update operation
     * @param appIds an array of registered appId's to sync, default of empty array will sync all apps
     * @param async whether to run updates in parallel or not (true by default)
     * @param callback a [SyncCallback] to notify on each app sync complete and when sync is complete
     */
    @JvmStatic
    @JvmOverloads
    fun sync(
        context: Context, appIds: Array<String> = arrayOf(), async: Boolean = true, callback: SyncCallback? = null
    ) {
        val syncApps: Map<String, LiveUpdate> = if (appIds.isEmpty()) {
            // Sync all apps
            instances
        } else {
            // Sync only provided appIds
            appArrayToInstances(appIds, callback)
        }

        // syncApps could be empty if no registered apps matched the provided appIds
        if (syncApps.isEmpty()) {
            Logger.error("Sync did not complete because there were no apps to be synced. " +
                    "Either there are no registered apps or the provided appIds were not registered.")
            callback?.onSyncComplete()
            return
        }

        liveUpdateScope.launch {
            if (async) {
                // Async mode will sync all provided appIds in parallel
                withContext(liveUpdateScope.coroutineContext) {
                    syncApps.map {
                        val appId = it.key
                        syncSemaphore.acquire()
                        val appJob = syncJobs[appId]
                        if (appJob == null || !appJob.isActive) {
                            liveUpdateIOScope.launch {
                                val job = this.coroutineContext.job
                                syncJobs[appId] = job
                                syncSemaphore.release()

                                try {
                                    syncApp(context, appId, callback)
                                } catch (e: CancellationException) {
                                    Logger.debug("Update task canceled for $appId")
                                    val liveUpdateInstance = instances[appId]
                                    liveUpdateInstance?.appState = AppState.CANCELED

                                    mainScope.launch {
                                        callback?.onAppComplete(
                                            FailResult(it.value, FailStep.CANCEL, "Sync canceled.")
                                        )
                                    }
                                }

                                syncSemaphore.acquire()
                                syncJobs.remove(appId)
                                syncSemaphore.release()
                            }
                        } else {
                            Logger.debug("Skipped adding sync task for $appId, one in progress!")
                            syncSemaphore.release()

                            mainScope.launch {
                                callback?.onAppComplete(
                                    FailResult(it.value, FailStep.CHECK, "Sync already in progress.")
                                )
                            }
                        }
                    }
                }.joinAll()
            } else {
                // Non-Async mode will sync all provided appIds sequentially
                syncApps.map {
                    val appId = it.key
                    syncSemaphore.acquire()
                    val appJob = syncJobs[appId]
                    if (appJob == null || !appJob.isActive) {
                        try {
                            withContext(liveUpdateIOScope.coroutineContext) {
                                val job = this.coroutineContext.job
                                syncJobs[appId] = job
                                syncSemaphore.release()

                                syncApp(context, appId, callback)
                            }
                        } catch (e: CancellationException) {
                            Logger.debug("Update task canceled for $appId")
                            val liveUpdateInstance = instances[appId]
                            liveUpdateInstance?.appState = AppState.CANCELED
                            mainScope.launch {
                                callback?.onAppComplete(
                                    FailResult(it.value, FailStep.CANCEL, "Sync canceled.")
                                )
                            }

                            // If we want to cancel the full sync, break loop here.
                        }

                        syncSemaphore.acquire()
                        syncJobs.remove(appId)
                        syncSemaphore.release()
                    } else {
                        Logger.debug("Skipped adding sync task for $appId, one in progress!")
                        syncSemaphore.release()
                        mainScope.launch {
                            callback?.onAppComplete(
                                FailResult(it.value, FailStep.CHECK, "Sync already in progress.")
                            )
                        }
                    }
                }
            }

            mainScope.launch {
                callback?.onSyncComplete()
            }
        }
    }

    private suspend fun syncApp(context: Context, appId: String, callback: SyncCallback? = null) {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Logger.error("App ID $appId not setup for live updates!")
            return
        }

        // Check for updates
        val checkResponse = checkForUpdates(context, liveUpdateInstance)
        if (checkResponse == null) {
            mainScope.launch {
                callback?.onAppComplete(FailResult(liveUpdateInstance, FailStep.CANCEL, "Sync canceled."))
            }
            return
        }

        if (checkResponse.error != null) {
            mainScope.launch {
                callback?.onAppComplete(FailResult(liveUpdateInstance, FailStep.CHECK, checkResponse.error.error.message))
            }
            return
        }

        coroutineContext.ensureActive()

        // Download available update
        val bodyData = checkResponse.success?.data
        val available = bodyData?.available
        if (available == true) {
            val snapshotId = checkResponse.success.data.snapshot
            val buildId = checkResponse.success.data.build

            if (snapshotId == null || buildId == null) {
                mainScope.launch {
                    callback?.onAppComplete(
                        FailResult(
                            liveUpdateInstance,
                            FailStep.CHECK,
                            "App snapshotId or buildId was null. Download can't continue."
                        )
                    )
                }
                return
            }

            var updateSource = Source.DOWNLOAD

            // Check cache before downloading
            Logger.debug("Update available for app $appId. Checking cache for ${snapshotId}...")
            val appSnapshots = getSnapshots(context)
            val snapshotExists = appSnapshots.filter { it.id == snapshotId }
            if (snapshotExists.isEmpty() || !getSnapshotDirectory(context, appId, snapshotId).exists()) {
                // Download
                Logger.debug("Snapshot not present in cache for $appId. Downloading snapshot ${snapshotId}...")

                if (bodyData.url.isNullOrEmpty()) {
                    mainScope.launch {
                        callback?.onAppComplete(
                            FailResult(
                                liveUpdateInstance,
                                FailStep.CHECK,
                                "No download location in API response. Download can't continue."
                            )
                        )
                    }
                    return
                }

                /**
                 * If the app is configured to use differential live updates, the sync process will
                 * download only new or changed files required for the update. Otherwise, a zipped
                 * whole update will be downloaded and extracted.
                 */
                if (liveUpdateInstance.updateStrategy == Strategy.DIFFERENTIAL) {
                    val downloadResult = downloadDifferentials(context, liveUpdateInstance, snapshotId, bodyData.url)
                    if (downloadResult != null) {
                        mainScope.launch {
                            // If the result is not null it is a failure. Report it to the callback.
                            callback?.onAppComplete(downloadResult)
                        }
                        return
                    }

                    coroutineContext.ensureActive()

                    // Verify against the bundled public key if the app uses Secure Live Updates
                    if (liveUpdateInstance.usesSecureLiveUpdates) {
                        val verified = verifyManifest(
                            context, liveUpdateInstance, getSnapshotDirectory(context, appId, snapshotId)
                        )
                        if (!verified) {
                            mainScope.launch {
                                callback?.onAppComplete(
                                    FailResult(
                                        liveUpdateInstance, FailStep.VERIFY, "Secure Live Update Verification failed."
                                    )
                                )
                            }
                            return
                        }
                    }
                } else {
                    // Zip
                    val zipFile = download(context, liveUpdateInstance, snapshotId, bodyData.url)
                    if (zipFile == null) {
                        mainScope.launch {
                            callback?.onAppComplete(
                                FailResult(liveUpdateInstance, FailStep.DOWNLOAD, "File download failed.")
                            )
                        }
                        return
                    }

                    coroutineContext.ensureActive()

                    // Unzip
                    val prepared = prepareSnapshot(liveUpdateInstance, zipFile)
                    if (prepared == null) {
                        mainScope.launch {
                            callback?.onAppComplete(
                                FailResult(liveUpdateInstance, FailStep.UNPACK, "Unzip failed.")
                            )
                        }
                        return
                    }


                    // Verify against the bundled public key if the app uses Secure Live Updates
                    if (liveUpdateInstance.usesSecureLiveUpdates) {
                        val verified = verifyManifest(context, liveUpdateInstance, prepared)
                        if (!verified) {
                            mainScope.launch {
                                callback?.onAppComplete(
                                    FailResult(
                                        liveUpdateInstance, FailStep.VERIFY, "Secure Live Update Verification failed."
                                    )
                                )
                            }
                            return
                        }
                    }

                    // Cleanup the zip file
                    zipFile.delete()
                }
            } else {
                updateSource = Source.CACHE
                Logger.debug("Cache for app $appId contains snapshot ${snapshotId}, reapplying...")
            }

            // Update
            val updateSuccessful = applyUpdate(context, liveUpdateInstance, snapshotId, buildId)
            if (!updateSuccessful) {
                mainScope.launch {
                    callback?.onAppComplete(
                        FailResult(liveUpdateInstance, FailStep.UPDATE, "Update failed.")
                    )
                }
                return
            }

            DataManager.updateLastSync(context, appId)

            val appDir = getLatestAppDirectory(context, appId)
            if (appDir != null) {
                Logger.debug("App $appId path updated to: ${appDir.path}")
            }

            val currentSnapshot = DataManager.getSnapshot(context, snapshotId)
            mainScope.launch {
                callback?.onAppComplete(
                    SyncResult(liveUpdateInstance, currentSnapshot, updateSource, true)
                )
            }
        } else {
            DataManager.updateLastSync(context, appId)
            liveUpdateInstance.appState = AppState.CHECKED
            liveUpdateInstance.availableUpdate = null
            Logger.debug("No update available for app $appId.")

            // Get current snapshot in use
            val app = DataManager.getApp(context, appId)
            if (app != null) {
                val currentSnapshot = app.getCurrentSnapshot(liveUpdateInstance.channelName)
                if (currentSnapshot != null) {
                    val snapshot = DataManager.getSnapshot(context, currentSnapshot)
                    if (snapshot != null) {
                        mainScope.launch {
                            callback?.onAppComplete(
                                SyncResult(liveUpdateInstance, snapshot, Source.CACHE, false)
                            )
                        }
                        return
                    } else {
                        callback?.onAppComplete(
                            FailResult(
                                liveUpdateInstance,
                                FailStep.UPDATE,
                                "No update available, but problem retrieving last update details."
                            )
                        )
                    }
                } else {
                    // No update, never updated.
                    callback?.onAppComplete(
                        SyncResult(liveUpdateInstance, null, Source.CACHE, false)
                    )
                }
            }

            // No update available and no app data previously saved in sharedprefs
            callback?.onAppComplete(
                SyncResult(liveUpdateInstance, null, Source.CACHE, false)
            )
        }
    }

    /**
     *  Attempts to cancel a running sync job
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.cancelSync("appId")
     * ```
     *
     * Example usage (Java):
     * ```java
     * LiveUpdateManager.cancelSync("appId");
     * ```
     *
     * @param appId the ID of the app associated with a sync job to cancel
     */
    @JvmStatic
    fun cancelSync(appId: String) {
        val job = syncJobs[appId]
        job?.cancel()
    }

    /**
     * Cancel all running sync jobs
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.cancelSync()
     * ```
     *
     * Example usage (Java):
     * ```java
     * LiveUpdateManager.cancelSync();
     * ```
     */
    @JvmStatic
    fun cancelSync() {
        syncJobs.map { syncJob ->
            syncJob.value.cancel()
            Logger.debug("Canceling job ${syncJob.key}")
        }
    }

    /**
     * Clean up any stale app versions.
     * Stale versions are any app snapshot files built for previous versions of the app binary and
     * not currently used by any app channel.
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.cleanStaleVersions(context, "appId")
     * ```
     *
     * Example usage (Java):
     * ```java
     * LiveUpdateManager.cleanStaleVersions(context, "appId");
     * ```
     *
     * @param context an Android [Context] used during the stale app cleanup operation
     * @param appId the ID of the registered app to clean up
     */
    @JvmStatic
    fun cleanStaleVersions(context: Context, appId: String) {
        // Get app binary version
        val manager = context.packageManager
        val info = manager.getPackageInfo(context.packageName, 0)
        val binaryVersion = info.versionName

        val app = DataManager.getApp(context, appId)
        if (app != null) {
            // Keep a cache of any snapshot that is currently used by a channel in case it needs
            // to be used again.
            val snapshotsInUse = mutableListOf<String>()

            val channels = app.channels
            channels.map { channel ->
                val channelSnapshot = channel.currentSnapshot
                snapshotsInUse.add(channelSnapshot)

                val snapshotObj = DataManager.getSnapshot(context, channel.currentSnapshot)
                if (snapshotObj != null && snapshotObj.binaryVersion != binaryVersion) {
                    Logger.debug(
                        "Update ${channel.currentSnapshot} for app ${app.id} was built for a different binary version (v${snapshotObj.binaryVersion}) caching..."
                    )
                    channel.currentSnapshot = ""
                }
            }

            val snapshotsToRemove = mutableListOf<String>()
            val appSnapshots = app.snapshots.toMutableList()
            appSnapshots.removeAll(snapshotsInUse)
            appSnapshots.map { snapshotId ->
                val snapshotObj = DataManager.getSnapshot(context, snapshotId)
                if (snapshotObj != null && snapshotObj.binaryVersion != binaryVersion) {
                    Logger.debug(
                        "Update $snapshotId for app ${app.id} was built for a different binary version (v${snapshotObj.binaryVersion}) and no longer used, removing from device"
                    )
                    snapshotsToRemove.add(snapshotId)
                    removeSnapshot(context, app.id, snapshotId)
                }
            }

            app.snapshots.removeAll(snapshotsToRemove)

            DataManager.saveApp(context, app)
        }
    }

    /**
     * Clean up all unused/old app versions.
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.cleanVersions(context)
     * ```
     *
     * Example usage (Java):
     * ```java
     * LiveUpdateManager.cleanVersions(context);
     * ```
     *
     * @param context an Android [Context] used during the app version cleanup operation
     */
    @JvmStatic
    fun cleanVersions(context: Context) {
        instances.map { app ->
            cleanVersions(context, app.key)
        }
    }

    /**
     * Clean up unused/old app versions related to a specific app.
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.cleanVersions(context, "appId")
     * ```
     *
     * Example usage (Java):
     * ```java
     * LiveUpdateManager.cleanVersions(context, "appId");
     * ```
     *
     * @param context an Android [Context] used during the app version cleanup operation
     * @param appId the ID of the app to clean
     */
    @JvmStatic
    fun cleanVersions(context: Context, appId: String) {
        // Clean stale versions first
        cleanStaleVersions(context, appId)

        val app = DataManager.getApp(context, appId)
        if (app != null) {
            val snapshotsInUse = mutableListOf<String>()
            val snapshotsToRemove = mutableListOf<String>()

            // Get snapshots in use we don't want to remove
            app.channels.map { channel ->
                snapshotsInUse.add(channel.currentSnapshot)
            }

            // Order remaining snapshots by last used date and remove any over the maxVersions cap
            val remainingSnapshots = app.snapshots.toMutableList()
            remainingSnapshots.removeAll(snapshotsInUse)
            remainingSnapshots.sortByDescending {
                val snapshotObj = DataManager.getSnapshot(context, it)
                snapshotObj?.lastUsed
            }

            val maxIndex = maxOf(remainingSnapshots.size, maxVersions)
            val trimSnapshots = remainingSnapshots.slice(maxVersions until maxIndex)
            trimSnapshots.map { snapshot ->
                Logger.debug("Cleanup - Removing excess app $appId snapshot $snapshot")
                snapshotsToRemove.add(snapshot)
                removeSnapshot(context, appId, snapshot)
            }

            app.snapshots.removeAll(snapshotsToRemove)
            DataManager.saveApp(context, app)
        }
    }

    /**
     * Synchronous function to check for an update for an app.
     *
     * Example usage (kotlin):
     * ```kotlin
     * val checkScope = CoroutineScope(Dispatchers.IO)
     * checkScope.launch {
     *     val response = LiveUpdateManager.checkForUpdate(context, "appId")
     *     println(response.body.toString())
     * }
     * ```
     *
     * Example usage (Java):
     * ```java
     * Executor executor = Executors.newSingleThreadExecutor();
     * executor.execute(() -> {
     *     CheckResponse response = LiveUpdateManager.checkForUpdate(getContext(), "appId");
     *     // Do something with response
     * });
     * ```
     *
     * @param context an Android [Context] used during the update check
     * @param appId the ID of the registered app to check
     * @return the [CheckResponse] result
     */
    @JvmStatic
    @Deprecated("No longer supported", ReplaceWith("sync"), DeprecationLevel.WARNING)
    fun checkForUpdate(context: Context, appId: String): CheckResponse? {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Logger.error("App ID $appId not setup for live updates!")
            return null
        }

        return checkForUpdates(context, liveUpdateInstance)
    }

    /**
     * Asynchronous function to check for an update for an app.
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.checkForUpdate(context, "appId", object : CheckCallback {
     *     override fun onComplete(result: CheckResponse?) {
     *         // Do something with the result
     *     }
     * })
     * ```
     *
     * Example usage (Java):
     * ```java
     * LiveUpdateManager.checkForUpdate(context, "appId", checkResponse -> {
     *     // Do something with response
     * });
     * ```
     *
     * @param context an Android [Context] used during the update check
     * @param appId the ID of the registered app to check
     * @param callback a [CheckCallback] to handle the response from the update check request
     */
    @JvmStatic
    @Deprecated("No longer supported", ReplaceWith("sync"), DeprecationLevel.WARNING)
    fun checkForUpdate(context: Context, appId: String, callback: CheckCallback) {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Logger.error("App ID $appId not setup for live updates!")
            callback.onComplete(null)
            return
        }

        liveUpdateIOScope.launch {
            val response = checkForUpdates(context, liveUpdateInstance)
            callback.onComplete(response)
        }
    }

    /**
     * Synchronous function to download an update for an app.
     *
     * Example usage (kotlin):
     * ```kotlin
     * val checkScope = CoroutineScope(Dispatchers.IO)
     * checkScope.launch {
     *     val response = LiveUpdateManager.downloadUpdate(context, "appId", "snapshotId")
     *     // Do something with result
     *     println(response.body.toString())
     * }
     * ```
     *
     * Example usage (Java):
     * ```java
     * Executor executor = Executors.newSingleThreadExecutor();
     * executor.execute(() -> {
     *     DownloadResponse response = LiveUpdateManager.downloadUpdate(getContext(), "appId", "snapshotId");
     *     // Do something with result
     * });
     * ```
     *
     * @param context an Android [Context] used during the download process
     * @param appId the ID of the registered app
     * @param snapshotId the ID of the snapshot to download
     * @return the [DownloadResponse] result
     */
    @JvmStatic
    @Deprecated("No longer supported", ReplaceWith("sync"), DeprecationLevel.WARNING)
    fun downloadUpdate(context: Context, appId: String, snapshotId: String): DownloadResponse? {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Logger.error("App ID $appId not setup for live updates!")
            return null
        }

        return Client.downloadUpdate(context, appId, snapshotId)
    }

    /**
     * Asynchronous function to download an update for an app.
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.downloadUpdate(applicationContext, "appId", "snapshotId", object : DownloadCallback {
     *     override fun onComplete(result: DownloadResponse?) {
     *         // Do something with result
     *     }
     * })
     * ```
     *
     * Example usage (Java):
     * ```java
     * LiveUpdateManager.downloadUpdate(getContext(), "appId", "snapshotId", downloadResponse -> {
     *     // Do something with response
     * });
     * ```
     *
     * @param context an Android [Context] used during the download process
     * @param appId the ID of the registered app
     * @param snapshotId the ID of the snapshot to download
     * @param callback a [DownloadCallback] to handle the response from the download request
     */
    @JvmStatic
    @Deprecated("No longer supported", ReplaceWith("sync"), DeprecationLevel.WARNING)
    fun downloadUpdate(
        context: Context, appId: String, snapshotId: String, callback: DownloadCallback
    ) {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Logger.error("App ID $appId not setup for live updates!")
            callback.onComplete(null)
            return
        }

        liveUpdateIOScope.launch {
            val response = Client.downloadUpdate(context, appId, snapshotId)
            callback.onComplete(response)
        }
    }

    /**
     * Synchronous function to extract a downloaded zipped app.
     *
     * Example usage (kotlin):
     * ```kotlin
     * val checkScope = CoroutineScope(Dispatchers.IO)
     * checkScope.launch {
     *     val response = LiveUpdateManager.extractUpdate("appId", zipFile)
     *     println(response?.path)
     * }
     * ```
     *
     * Example usage (Java):
     * ```java
     * Executor executor = Executors.newSingleThreadExecutor();
     * executor.execute(() -> {
     *     File extractedFile = LiveUpdateManager.extractUpdate(getContext(), zipFile);
     * });
     * ```
     *
     * @param appId the ID of the registered app
     * @param zipFile the zipped app download to extract
     * @return a file directory where the app was unzipped
     */
    @JvmStatic
    @Deprecated("No longer supported", ReplaceWith("sync"), DeprecationLevel.WARNING)
    fun extractUpdate(appId: String, zipFile: File): File? {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Logger.error("App ID $appId not setup for live updates!")
            return null
        }

        return prepareSnapshot(liveUpdateInstance, zipFile)
    }

    /**
     * Asynchronous function to extract a downloaded zipped app.
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.extractUpdate("appId", zipFile, object : ExtractCallback {
     *     override fun onComplete(result: File?) {
     *         println(response?.path)
     *     }
     * })
     * ```
     *
     * Example usage (Java):
     * ```java
     * LiveUpdateManager.extractUpdate(getContext(), zipFile, extractedFile -> {
     *     // Do something with file
     * });
     * ```
     *
     * @param appId the ID of the registered app
     * @param zipFile the zipped app download to extract
     * @param callback an [ExtractCallback] to handle the response from the extract request
     */
    @JvmStatic
    @Deprecated("No longer supported", ReplaceWith("sync"), DeprecationLevel.WARNING)
    fun extractUpdate(appId: String, zipFile: File, callback: ExtractCallback) {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Logger.error("App ID $appId not setup for live updates!")
            callback.onComplete(null)
            return
        }

        liveUpdateIOScope.launch {
            val file = prepareSnapshot(liveUpdateInstance, zipFile)
            callback.onComplete(file)
        }
    }

    /**
     * Verify a downloaded Live Update if it is configured to use Secure Live Updates.
     *
     * Example usage (kotlin):
     * ```kotlin
     * val verified = LiveUpdateManager.verifyUpdate(context, "appId", liveUpdateFiles)
     * ```
     *
     * Example usage (Java):
     * ```java
     * boolean verified = LiveUpdateManager.verifyUpdate(context, "appId", liveUpdateFiles);
     * ```
     *
     * @param context an Android [Context] used during the verification process
     * @param appId the ID of the app to verify
     * @param updatePath the path of the downloaded and extracted Live Update
     * @return true if the app is verified or the app is not setup for Secure Live Updates, false if
     * not setup for Live Updates or cannot be verified
     */
    @JvmStatic
    @Deprecated("No longer supported", ReplaceWith("sync"), DeprecationLevel.WARNING)
    fun verifyUpdate(context: Context, appId: String, updatePath: File): Boolean {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Logger.error("App ID $appId not setup for live updates!")
            return false
        }

        if (!liveUpdateInstance.usesSecureLiveUpdates) {
            Logger.info("App ID $appId is not configured to use Secure Live Updates.")
            return true
        }

        return verifyManifest(context, liveUpdateInstance, updatePath)
    }

    /**
     * Apply a downloaded and extracted Live Update snapshot to the portal for next load.
     *
     * Example usage (kotlin):
     * ```kotlin
     * LiveUpdateManager.applyUpdate(context, "appId", "snapshotId", "buildId")
     * ```
     *
     * Example usage (Java):
     * ```java
     * LiveUpdateManager.applyUpdate(context, "appId", "snapshotId", "buildId");
     * ```
     *
     * @param context an Android [Context] used when applying the Live Update
     * @param appId the ID of the app to apply a snapshot update to
     * @param snapshotId the ID of the snapshot to apply
     * @param buildId the ID of the build to apply
     */
    @JvmStatic
    @Deprecated("No longer supported", ReplaceWith("sync"), DeprecationLevel.WARNING)
    fun applyUpdate(context: Context, appId: String, snapshotId: String, buildId: String) {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Logger.error("App ID $appId not setup for live updates!")
            return
        }

        applyUpdate(context, liveUpdateInstance, snapshotId, buildId)
    }

    /**
     * Get the map of all registered apps using Live Updates.
     *
     * Example usage (kotlin):
     * ```kotlin
     * val apps = LiveUpdateManager.getApps()
     * ```
     *
     * Example usage (Java):
     * ```java
     * Map<String, LiveUpdate> apps = LiveUpdateManager.getApps();
     * ```
     *
     * @return a map of all registered apps that use Live Updates.
     */
    @JvmStatic
    fun getApps(): MutableMap<String, LiveUpdate> {
        return instances
    }

    /**
     * Get the latest directory for the updated app resources, if available. If the app has
     * not been updated through Live Updates, null will be returned.
     *
     * Example usage (kotlin):
     * ```kotlin
     * val latestAppDirectory = LiveUpdateManager.getLatestAppDirectory(context, "appId")
     * ```
     *
     * Example usage (Java):
     * ```java
     * File latestAppDirectory = LiveUpdateManager.getLatestAppDirectory(context, "appId");
     * ```
     *
     * @param context an Android [Context] used to get the latest app directory
     * @param appId the ID of the app directory to retrieve
     * @return directory file if available, otherwise null
     */
    @JvmStatic
    fun getLatestAppDirectory(context: Context, appId: String): File? {
        val instance = instances[appId]
        if (instance != null) {
            val app = DataManager.getApp(context, appId)
            if (app != null) {
                val channel = app.getChannel(instance.channelName)
                if (channel != null) {
                    if (channel.currentSnapshot.isNotEmpty()) {
                        return getSnapshotDirectory(context, appId, channel.currentSnapshot)
                    }
                }
            }
        }

        return null
    }

    internal fun getSnapshots(context: Context): List<Snapshot> {
        return DataManager.getSnapshots(context)
    }

    internal fun getLiveUpdatesDirectory(context: Context): File {
        return File(context.filesDir, LIVE_UPDATES_DIR)
    }

    private fun getSnapshotDirectory(context: Context, appId: String, snapshotId: String): File {
        val filePath = context.filesDir.path + separator + LIVE_UPDATES_DIR + separator + appId + separator + snapshotId
        return File(filePath)
    }

    private fun appArrayToInstances(
        appIds: Array<String>, callback: SyncCallback? = null
    ): MutableMap<String, LiveUpdate> {
        val map: MutableMap<String, LiveUpdate> = mutableMapOf()
        appIds.map {
            val instance = instances[it]
            if (instance != null) {
                map[it] = instance
            } else {
                Logger.error("appId $it was not found in the registered app list.")
                callback?.onAppComplete(FailResult(LiveUpdate(it), FailStep.CHECK, "App was not registered."))
            }
        }

        return map
    }

    private fun checkForUpdates(context: Context, instance: LiveUpdate): CheckResponse? {
        instance.appState = AppState.CHECKING
        instance.availableUpdate = null

        val checkRequest = DataManager.getCheckUpdateData(context, instance) ?: return null

        val responseBody = Client.checkForUpdate(checkRequest)
        if (responseBody != null) {
            if (responseBody.error != null) {
                Logger.error(responseBody.error.toString())
                instance.appState = AppState.FAILED
                instance.availableUpdate = null
            } else {
                val bodyData = responseBody.success?.data
                val available = bodyData?.available
                if (available == true) {
                    instance.appState = AppState.AVAILABLE
                    instance.availableUpdate = AvailableUpdateState.AVAILABLE
                }
            }
        }

        return responseBody
    }

    private fun download(context: Context, instance: LiveUpdate, snapshotId: String, downloadURL: String): File? {
        instance.appState = AppState.DOWNLOADING

        val download = Client.downloadUpdate(context, instance.appId, snapshotId, downloadURL)
        if (download.error != null) {
            instance.appState = AppState.FAILED
            Logger.error("Error downloading update for app ${instance.appId} snapshot $snapshotId.")
            return null
        }

        download.file?.let {
            Logger.debug("The saved ${it.path} file is ${it.length()} bytes")
            instance.appState = AppState.DOWNLOADED
            instance.availableUpdate = AvailableUpdateState.PENDING
            return it
        }

        return null
    }

    private suspend fun downloadDifferentials(
        context: Context, instance: LiveUpdate, snapshotId: String, downloadURL: String
    ): FailResult? = withContext(Dispatchers.IO) {
        instance.appState = AppState.DOWNLOADING

        Logger.debug("Differential download process started")

        // Get manifest
        val downloadedManifest = Client.getManifest(instance.appId, snapshotId, downloadURL)
        if (downloadedManifest.error != null) {
            return@withContext FailResult(
                instance,
                FailStep.DOWNLOAD,
                "${downloadedManifest.error.error.type}: ${downloadedManifest.error.error.message}. File download failed."
            )
        }

        if (downloadedManifest.files.isNullOrEmpty()) {
            return@withContext FailResult(
                instance,
                FailStep.DOWNLOAD,
                "Manifest from Appflow does not contain any files to download. File download failed."
            )
        }

        if (downloadedManifest.responseURL == null || downloadedManifest.manifestHref == null) {
            return@withContext FailResult(
                instance,
                FailStep.DOWNLOAD,
                "Manifest from Appflow does not contain a valid responseURL or manifestHref. File download failed."
            )
        }

        val appDir = File(getLiveUpdatesDirectory(context), instance.appId)

        // Make dirs
        val tempDirPath = "tmp_$snapshotId"
        val tempDir = setupSnapshotDirectory(context, instance.appId, tempDirPath)

        // Download a hard copy of the manifest to the new tmp app directory
        val manifestFileURL = "${downloadedManifest.responseURL}/${downloadedManifest.manifestHref}"
        Client.downloadFile(
            instance.appId,
            snapshotId,
            downloadedManifest.manifestHref,
            manifestFileURL,
            downloadedManifest.responseQuery,
            File(appDir, tempDirPath)
        )

        // Check if there is a previous live update
        val currentLiveUpdateDir = getLatestAppDirectory(context, instance.appId)
        var shouldSkipAssetCheck = false

        /**
         * Check if there is a previous live update. If there is, we want to try and use its file contents to do a
         * differential update.
         */
        if (currentLiveUpdateDir != null) {
            // Ensure the latest update contains the required manifest file in order to do a diff update
            if(File(currentLiveUpdateDir.path + separator + SECURE_LIVE_UPDATES_MANIFEST).exists()) {
                Logger.debug("Attempting differential update from a previous live update $currentLiveUpdateDir")

                // Get copy of manifest from the last update
                val diffDownloadFiles: Set<PayloadFile>
                val diffCopyFiles: Set<PayloadFile>
                try {
                    val currentManifestFile = File(currentLiveUpdateDir.path + separator + SECURE_LIVE_UPDATES_MANIFEST)
                    val currentManifest = Json.decodeFromString(Manifest.serializer(), currentManifestFile.readText(Charsets.UTF_8))

                    val currentFiles = currentManifest.decodedPayload.toSet()
                    val downloadFiles = downloadedManifest.files.toSet()
                    diffDownloadFiles = downloadFiles.minus(currentFiles)
                    diffCopyFiles = currentFiles.intersect(downloadFiles)
                } catch (e: Exception) {
                    // Delete the directory and cleanup because the download was unsuccessful
                    tempDir.deleteRecursively()

                    Logger.error("Problem opening or deserializing the previous live update manifest")
                    Logger.error(e.toString())
                    return@withContext FailResult(
                        instance,
                        FailStep.DOWNLOAD,
                        "Live update manifest path broken or file malformed. File download failed."
                    )
                }

                // Download files and continue when finished all of them
                val downloadErrors = downloadFiles(
                    diffDownloadFiles,
                    instance,
                    snapshotId,
                    downloadedManifest,
                    appDir,
                    tempDirPath
                )

                if (downloadErrors.isNotEmpty()) {
                    Logger.error("Problems downloading files for ")
                    Logger.error(downloadErrors.toString())

                    // Delete the directory and cleanup because the download was unsuccessful
                    tempDir.deleteRecursively()

                    return@withContext FailResult(
                        instance,
                        FailStep.DOWNLOAD,
                        "${downloadErrors.size} file(s) were unable to be downloaded. File download failed."
                    )
                }

                instance.appState = AppState.DOWNLOADED

                // Copy the files that are the same between updates
                instance.appState = AppState.COPYING
                val copyErrors = copyFiles(diffCopyFiles, currentLiveUpdateDir, appDir, tempDirPath)
                if (copyErrors.isNotEmpty()) {
                    // Delete the directory and cleanup because the copy was unsuccessful
                    tempDir.deleteRecursively()

                    return@withContext FailResult(
                        instance, FailStep.COPY, "${copyErrors.size} file(s) were unable to be copied. File copy failed."
                    )
                } else {
                    // Partial diff download and copy successful. Rename tmp snapshot dir.
                    tempDir.renameTo(File(appDir, snapshotId))

                    Logger.debug("Differential update from a previous live update was successful")
                    instance.appState = AppState.COPIED
                    return@withContext null
                }
            }

            // There was no manifest in the previous update. Skip the next check and do a full download
            shouldSkipAssetCheck = true
        }

        /**
         * No previous live update, check if there is a manifest in the bundled web assets
         * This is skipped in lieu of a full download if problems prevented a diff update in the previous step
         */
        if (!shouldSkipAssetCheck) {
            val originalAssetsPath = instance.assetPath
            if (!originalAssetsPath.isNullOrEmpty()) {
                val assetsList = context.assets.list(originalAssetsPath)
                if (!assetsList.isNullOrEmpty()) {
                    // Check if the previous update contains a manifest
                    if (assetsList.contains(SECURE_LIVE_UPDATES_MANIFEST)) {
                        /*
                         * At this point we confirm the original files exist and a manifest is present.
                         * We can attempt a diff.
                         */
                        Logger.debug("Attempting differential update from bundled web assets")

                        // Get copy of manifest from the original bundle
                        val diffDownloadFiles: Set<PayloadFile>
                        val diffCopyFiles: Set<PayloadFile>
                        try {
                            val originManifestFile = context.assets.open(originalAssetsPath + separator + SECURE_LIVE_UPDATES_MANIFEST)
                            val originManifest = Json.decodeFromStream<Manifest>(originManifestFile)

                            val originFiles = originManifest.decodedPayload.toSet()
                            val downloadFiles = downloadedManifest.files.toSet()
                            diffDownloadFiles = downloadFiles.minus(originFiles)
                            diffCopyFiles = originFiles.intersect(downloadFiles)
                        } catch (e: Exception) {
                            // Delete the directory and cleanup because the download was unsuccessful
                            tempDir.deleteRecursively()

                            Logger.error("Problem opening or deserializing the bundled web assets live update manifest")
                            Logger.error(e.toString())
                            return@withContext FailResult(
                                instance,
                                FailStep.DOWNLOAD,
                                "Live update manifest path broken or file malformed. File download failed."
                            )
                        }

                        // Download files and continue when finished all of them
                        val downloadErrors = downloadFiles(
                            diffDownloadFiles,
                            instance,
                            snapshotId,
                            downloadedManifest,
                            appDir,
                            tempDirPath
                        )

                        if (downloadErrors.isNotEmpty()) {
                            Logger.error("Problems downloading files for ")
                            Logger.error(downloadErrors.toString())

                            // Delete the directory and cleanup because the download was unsuccessful
                            tempDir.deleteRecursively()

                            return@withContext FailResult(
                                instance,
                                FailStep.DOWNLOAD,
                                "${downloadErrors.size} file(s) were unable to be downloaded. File download failed."
                            )
                        }

                        instance.appState = AppState.DOWNLOADED

                        // Copy the files that are the same between updates
                        instance.appState = AppState.COPYING
                        val copyErrors = copyFilesFromAssets(context, diffCopyFiles, originalAssetsPath, appDir, tempDirPath)
                        if (copyErrors.isNotEmpty()) {
                            // Delete the directory and cleanup because the copy was unsuccessful
                            tempDir.deleteRecursively()

                            return@withContext FailResult(
                                instance, FailStep.COPY, "${copyErrors.size} file(s) were unable to be copied. File copy failed."
                            )
                        }

                        // Partial diff download and copy successful. Rename tmp snapshot dir.
                        tempDir.renameTo(File(appDir, snapshotId))

                        Logger.debug("Differential update from bundled web assets was successful")
                        instance.appState = AppState.COPIED
                        return@withContext null
                    }
                }
            }
        }

        // If we get to this point, there is no prior update or manifest in the bundled web assets, do a full download

        /*
         * Download a fresh full update.
         * There is either no previous update OR the previous update was not downloaded
         * using the differential method. A full update is needed in these conditions.
         */
        Logger.debug("Attempting a full download using differential live updates")
        val errors = downloadFiles(
            downloadedManifest.files.toSet(),
            instance,
            snapshotId,
            downloadedManifest,
            appDir,
            tempDirPath
        )

        if (errors.isNotEmpty()) {
            Logger.error("Problems downloading files for ")
            Logger.error(errors.toString())

            // Delete the directory and cleanup because the download was unsuccessful
            tempDir.deleteRecursively()

            return@withContext FailResult(
                instance,
                FailStep.DOWNLOAD,
                "${errors.size} file(s) were unable to be downloaded. File download failed."
            )
        } else {
            // Full app download successful. Rename tmp snapshot dir.
            tempDir.renameTo(File(appDir, snapshotId))

            Logger.debug("Full download using differential live updates was successful")
            instance.appState = AppState.DOWNLOADED
            return@withContext null
        }
    }

    private suspend fun downloadFiles(
        diffDownloadFiles: Set<PayloadFile>,
        instance: LiveUpdate,
        snapshotId: String,
        downloadedManifest: ManifestResponse,
        appDir: File,
        tempDirPath: String
    ) = withContext(Dispatchers.IO) {
        val deferredDownload = diffDownloadFiles.map {
            async(Dispatchers.IO) {
                try {
                    val fileURL = "${downloadedManifest.responseURL}/${it.href}"
                    Client.downloadFile(
                        instance.appId,
                        snapshotId,
                        it.href,
                        fileURL,
                        downloadedManifest.responseQuery,
                        File(appDir, tempDirPath)
                    )
                } catch (e: Exception) {
                    null
                }
            }
        }
        val listOfReturnData = deferredDownload.awaitAll()
        return@withContext listOfReturnData.filterNotNull().filter { downloadResponse -> downloadResponse.error != null }
    }

    private suspend fun copyFiles(
        diffCopyFiles: Set<PayloadFile>,
        currentLiveUpdateDir: File,
        appDir: File,
        tempDirPath: String
    ) = withContext(Dispatchers.IO) {
        val copyErrors = mutableListOf<PayloadFile>()
        val deferredCopy = diffCopyFiles.map {
            async(Dispatchers.IO) {
                // Copy file from existing app/snapshot dir to the new tmp one
                Logger.debug(
                    "Copying existing web file from ${currentLiveUpdateDir.path + separator + it.href} to ${
                        File(
                            appDir,
                            tempDirPath + separator + it.href
                        ).absolutePath
                    }"
                )

                try {
                    val existingFile = File(currentLiveUpdateDir.path + separator + it.href)
                    existingFile.copyTo(File(appDir, tempDirPath + separator + it.href), true)
                } catch (e: Exception) {
                    copyErrors.add(it)
                    Logger.error("Copy error for file ${it.href}")
                    Logger.error(e.toString())
                }
            }
        }
        deferredCopy.awaitAll()
        return@withContext copyErrors
    }

    private suspend fun copyFilesFromAssets(
        context: Context,
        diffCopyFiles: Set<PayloadFile>,
        originalAssetsPath: String,
        appDir: File,
        tempDirPath: String
    ) = withContext(Dispatchers.IO) {
        val copyErrors = mutableListOf<PayloadFile>()
        val deferredCopy = diffCopyFiles.map {
            async(Dispatchers.IO) {
                // Copy file from existing app/snapshot dir to the new tmp one
                val copyToFile = File(appDir, tempDirPath + separator + it.href)
                Logger.debug(
                    "Copying existing web file from asset path ${originalAssetsPath + separator + it.href} to ${
                        copyToFile.absolutePath
                    }"
                )

                try {
                    context.assets.open(originalAssetsPath + separator + it.href).use { input ->
                        copyToFile.parentFile?.mkdirs()
                        copyToFile.createNewFile()
                        copyToFile.outputStream().use { output ->
                            input.copyTo(output)
                        }
                    }
                } catch (e: Exception) {
                    copyErrors.add(it)
                    Logger.error("Copy error for file ${it.href}")
                    Logger.error(e.toString())
                }
            }
        }
        deferredCopy.awaitAll()
        return@withContext copyErrors
    }

    private fun setupMainDirectory(context: Context, reset: Boolean = false) {
        val liveUpdateDir = getLiveUpdatesDirectory(context)

        // if resetting, delete the whole dir and start fresh
        if (reset) {
            liveUpdateDir.deleteRecursively()
        }

        // create live updates directory if it doesn't exist
        if (!liveUpdateDir.exists()) {
            liveUpdateDir.mkdir()
        }
    }

    private fun setupAppDirectory(context: Context, appId: String) {
        val appDir = File(getLiveUpdatesDirectory(context), appId)

        // create app directory if it doesn't exist
        if (!appDir.exists()) {
            appDir.mkdir()
        }
    }

    private fun setupSnapshotDirectory(context: Context, appId: String, snapshotId: String): File {
        // Setup app dir first if it doesn't exist
        setupAppDirectory(context, appId)

        val snapshotDir = File(File(getLiveUpdatesDirectory(context), appId), snapshotId)

        // create snapshot directory if it doesn't exist
        if (!snapshotDir.exists()) {
            snapshotDir.mkdir()
        }

        return snapshotDir
    }

    private fun prepareSnapshot(instance: LiveUpdate, file: File): File? {
        instance.appState = AppState.UNPACKING

        return try {
            val resultDir = ZipUtility.unzip(file)
            instance.appState = AppState.UNPACKED
            resultDir
        } catch (exception: IOException) {
            instance.appState = AppState.FAILED
            Logger.error("Error preparing downloaded update for app ${instance.appId} file ${file.path}.")
            Logger.error("Error unzipping ${file.path}!")
            Logger.error(exception.stackTraceToString())
            null
        }
    }

    private fun verifyManifest(context: Context, instance: LiveUpdate, updatePath: File): Boolean {
        instance.appState = AppState.VERIFYING

        val sluManifestPath = updatePath.path + separator + SECURE_LIVE_UPDATES_MANIFEST
        val sluManifest = File(sluManifestPath)
        if (!sluManifest.exists()) {
            Logger.error("Secure Live Update manifest file for app ${instance.appId} does not exist.")
            instance.appState = AppState.FAILED
            return false
        }

        try {
            val manifest = Json.decodeFromString(
                Manifest.serializer(), sluManifest.readText(Charsets.UTF_8)
            )

            if (manifest.signatures.isNullOrEmpty()) {
                Logger.error("Secure Live Update manifest file for app ${instance.appId} does not contain signatures.")
                instance.appState = AppState.FAILED
                return false
            }

            try {
                val pemParser = PEMParser(PemReader(context.assets.open(secureLiveUpdatePEM).bufferedReader()))
                val converter = JcaPEMKeyConverter()
                val publicKeyInfo = SubjectPublicKeyInfo.getInstance(pemParser.readObject())
                val pubKey = converter.getPublicKey(publicKeyInfo) as RSAPublicKey

                var verified = false
                val payload = manifest.payload.toByteArray(Charsets.UTF_8)
                for (signature in manifest.signatures) {
                    val header = signature.protected.toByteArray(Charsets.UTF_8)
                    val decodedSignature = Base64.decode(signature.signature, Base64.DEFAULT)
                    val rsaSignature = Signature.getInstance("SHA256withRSA")
                    rsaSignature.initVerify(pubKey)
                    rsaSignature.update(header)
                    rsaSignature.update('.'.code.toByte())
                    rsaSignature.update(payload)

                    verified = rsaSignature.verify(decodedSignature)
                    if (verified) {
                        break
                    }
                }

                if (!verified) {
                    Logger.error("Verification of Live Update app update ${instance.appId} failed.")
                    instance.appState = AppState.FAILED
                    return false
                }

                // Update is valid if zip signature and file checksums match
                return if (verifyHashes(updatePath, manifest)) {
                    Logger.info("Verification of Live Update succeeded.")
                    instance.appState = AppState.VERIFIED
                    true
                } else {
                    instance.appState = AppState.FAILED
                    false
                }
            } catch (e: Exception) {
                Logger.error("Problem validating Secure Live Updates manifest from app update ${instance.appId}.")
                Logger.error(sluManifest.path)
                Logger.error(e.toString())
            }
        } catch (e: Exception) {
            Logger.error("Problem reading Secure Live Updates manifest from app update ${instance.appId}.")
            Logger.error(sluManifest.path)
            Logger.error(e.toString())
        }

        instance.appState = AppState.FAILED
        return false
    }

    private fun verifyHashes(updatePath: File, manifest: Manifest): Boolean {
        var areFileChecksumsValid = true

        try {
            val payloadFiles = manifest.decodedPayload.mapNotNull {
                val verifyFile = File(updatePath.path + separator + it.href)
                if (verifyFile.exists()) {

                    val integrity = it.integrity
                    verifyFile.source().buffer().use { source ->
                        HashingSink.sha256(blackholeSink()).use { sink ->
                            source.readAll(sink)
                            val base64hash = sink.hash.base64()
                            if (!integrity.contains(base64hash)) {
                                Logger.error("Secure Live Updates verification error: checksum for file ${verifyFile.path} invalid.")
                                areFileChecksumsValid = false
                            }
                        }
                    }

                    verifyFile
                } else {
                    Logger.error("Secure Live Updates verification error: file ${verifyFile.path} not found.")
                    null
                }
            }.toHashSet()

            val diskFiles = updatePath.walk().mapNotNull {
                if (it.isDirectory || it.path == updatePath.path + separator + SECURE_LIVE_UPDATES_MANIFEST) {
                    null
                } else {
                    it
                }
            }.toHashSet()


            val extras = diskFiles subtract payloadFiles
            extras.forEach {
                Logger.error("Secure Live Updates verification error: file ${it.path} present in Live Update bundle but not in the manifest.")
            }

            return extras.isEmpty() && areFileChecksumsValid
        } catch (e: Exception) {
            Logger.error("Secure Live Updates verification error: manifest for ${updatePath.path} invalid.")
        }

        return false
    }

    private fun applyUpdate(
        context: Context, instance: LiveUpdate, snapshotId: String, buildId: String
    ): Boolean {
        instance.appState = AppState.UPDATING

        // Get app binary version
        val manager = context.packageManager
        val info = manager.getPackageInfo(context.packageName, 0)

        var currentSnapshot = DataManager.getSnapshot(context, snapshotId)
        if (currentSnapshot == null) {
            currentSnapshot = Snapshot(snapshotId, buildId, info.versionName, System.currentTimeMillis())
        } else {
            currentSnapshot.lastUsed = System.currentTimeMillis()
            currentSnapshot.binaryVersion = info.versionName
        }

        DataManager.addSnapshot(context, currentSnapshot)

        val app = DataManager.getApp(context, instance.appId)
        if (app != null) {
            val currentChannel = app.channels.filter { channel -> channel.id == instance.channelName }
            if (currentChannel.isNotEmpty()) {
                val channel = currentChannel[0]
                channel.currentSnapshot = snapshotId
                if (!app.snapshots.contains(snapshotId)) {
                    app.snapshots.add(snapshotId)
                }

                DataManager.saveApp(context, app)

                instance.appState = AppState.UPDATED
                instance.availableUpdate = AvailableUpdateState.READY
                return true
            } else {
                instance.appState = AppState.FAILED
                Logger.error("Error retrieving App channel data from local store.")
            }
        } else {
            instance.appState = AppState.FAILED
            Logger.error("Error retrieving App data from local store.")
        }

        return false
    }

    private fun removeSnapshot(context: Context, appId: String, snapshotId: String) {
        // remove from shared prefs
        DataManager.deleteSnapshot(context, snapshotId)

        // delete files
        val snapshotDir = getSnapshotDirectory(context, appId, snapshotId)
        if (snapshotDir.exists()) {
            snapshotDir.deleteRecursively()
        }
    }

    private fun storeAppData(context: Context, liveUpdate: LiveUpdate) {
        var app = DataManager.getApp(context, liveUpdate.appId)
        if (app == null) {
            app = App(liveUpdate.appId)
        }

        app.addChannel(Channel(liveUpdate.channelName))
        DataManager.saveApp(context, app)
    }
}