package io.ionic.liveupdates

import java.io.*
import java.util.zip.ZipFile

/**
 * The ZipUtility class is used to extract zipped Live Update payloads retrieved from [Appflow](https://ionic.io/appflow).
 */
object ZipUtility {
    private const val BUFFER_SIZE = 4096

    /**
     * Extracts a provided zip file to a directory of the same name
     *
     * @param zipFile a valid zip file to extract
     * @return the file directory of the extracted zip archive
     */
    @Throws(IOException::class)
    fun unzip(zipFile: File): File {
        val destDir = File(zipFile.parent, zipFile.nameWithoutExtension)
        if (!destDir.exists()) {
            destDir.mkdir()
        }

        ZipFile(zipFile).use { zip ->
            zip.entries().asSequence().forEach { entry ->
                zip.getInputStream(entry).use { input ->
                    val filePath = destDir.path + File.separator + entry.name

                    if (entry.isDirectory) {
                        // Make directory
                        val dir = File(filePath)
                        dir.mkdir()
                    } else {
                        // Extract file
                        extractFile(input, filePath)
                    }
                }
            }
        }

        return destDir
    }

    @Throws(IOException::class)
    private fun extractFile(inputStream: InputStream, destinationPath: String) {
        val outStream = BufferedOutputStream(FileOutputStream(destinationPath))
        val bytesIn = ByteArray(BUFFER_SIZE)
        var read: Int

        while (inputStream.read(bytesIn).also { read = it } != -1) {
            outStream.write(bytesIn, 0, read)
        }

        outStream.close()
    }
}