package io.ionic.liveupdates.network

import io.ionic.liveupdates.data.model.FailResult
import io.ionic.liveupdates.data.model.SyncResult

/**
 * A callback used to access the results from the Live Update sync process.
 *
 * Example usage (kotlin):
 * ```kotlin
 * LiveUpdateManager.sync(context, "appId", callback = object : SyncCallback {
 *     override fun onAppComplete(syncResult: SyncResult) {
 *         Log.d("LiveUpdate","CALLBACK: Sync success for app ${syncResult.liveUpdate.appId}!")
 *     }
 *
 *     override fun onAppComplete(failResult: FailResult) {
 *         Log.e("LiveUpdate","CALLBACK: Sync failed at step ${failResult.failStep.name} for app ${failResult.liveUpdate.appId}!")
 *     }
 *
 *     override fun onSyncComplete() {
 *         Log.d("LiveUpdate","CALLBACK: Sync finished!")
 *     }
 * })
 * ```
 *
 * Example usage (Java):
 * ```java
 * LiveUpdateManager.sync(this, "appId", new SyncCallback() {
 *     @Override
 *     public void onAppComplete(@NonNull SyncResult syncResult) {
 *         Log.d("LiveUpdate","CALLBACK: Sync success for app " + syncResult.getLiveUpdate().getAppId());
 *     }
 *
 *     @Override
 *     public void onAppComplete(@NonNull FailResult failResult) {
 *         Log.d("LiveUpdate","CALLBACK: Sync failed at step " + failResult.getFailStep().name() + " for app " + failResult.getLiveUpdate().getAppId());
 *     }
 *
 *     @Override
 *     public void onSyncComplete() {
 *         Log.d("LiveUpdate","CALLBACK: Sync finished!");
 *     }
 * });
 * ```
 */
interface SyncCallback {
    /**
     * Called when an application is finished syncing and completed successfully.
     *
     * @param syncResult the details of the sync
     */
    fun onAppComplete(syncResult: SyncResult)

    /**
     * Called when an application is finished syncing but it failed.
     *
     * @param failResult the details of the failed sync
     */
    fun onAppComplete(failResult: FailResult)

    /**
     * Called when the sync process for one or more apps is finished running.
     */
    fun onSyncComplete()
}