package io.ionic.liveupdates.data.model

import android.util.Base64
import kotlinx.serialization.Serializable
import kotlinx.serialization.builtins.ListSerializer
import kotlinx.serialization.json.Json

/**
 * A class representing a file within a Live Update payload.
 *
 * @property href the path of the file
 * @property size the size of the file
 * @property integrity the MD5 hash of the file
 */
@Serializable
data class PayloadFile(val href: String, val size: Long, val integrity: String)

/**
 * A class representing the signature in the secure manifest.
 *
 * @property protected the header for the payload
 * @property signature used to validate the payload is trustworthy
 */
@Serializable
data class Signature(val protected: String, val signature: String)

/**
 * A class representing the secure manifest associated with validating secured Live Updates payloads.
 *
 * @property signatures a list of signatures in the manifest
 * @property payload the encoded payload of the manifest
 */
@Serializable
data class SecureManifest(val signatures: List<Signature>, val payload: String) {
    /**
     * The decoded payload containing a list of [PayloadFile]
     */
    val decodedPayload: List<PayloadFile> by lazy {
        val decodedPayload = String(Base64.decode(payload, Base64.DEFAULT))
        Json.decodeFromString(ListSerializer(PayloadFile.serializer()), decodedPayload)
    }
}