package io.ionic.liveupdates.data.model

import kotlinx.serialization.Serializable

/**
 * A class that represents an application and its associated channel and downloaded snapshots.
 *
 * @property id the ID of the application
 */
@Serializable
data class App(val id: String) {
    /**
     * A list of associated channels linked to the app
     */
    val channels: MutableList<Channel> = mutableListOf()

    /**
     * A list of snapshots linked to the app
     */
    val snapshots: MutableList<String> = mutableListOf()

    /**
     * The timestamp the app was last synced
     */
    var lastSync: Long = -1

    /**
     * Adds a channel to the app.
     *
     * @param channel a [Channel] to add to the app
     */
    fun addChannel(channel: Channel) {
        if (!channels.contains(channel)) {
            channels.add(channel)
        }
    }

    /**
     * Retrieve the linked Channel object.
     *
     * @param id the ID of the channel
     * @return the [Channel] object if it is linked
     */
    fun getChannel(id: String): Channel? {
        val currentChannel = channels.filter { channel -> channel.id == id }
        if (currentChannel.isNotEmpty()) {
            return currentChannel[0]
        }

        return null
    }

    /**
     * Get the current snapshot at the head of the channel ID provided.
     *
     * @param channelId the ID of the channel to get the latest snapshot ID from
     * @return the ID of the latest [Snapshot] downloaded for the provided channel ID if one exists
     */
    fun getCurrentSnapshot(channelId: String): String? {
        val currentChannel = getChannel(channelId)
        if (currentChannel != null) {
            if (snapshots.contains(currentChannel.currentSnapshot)) {
                return currentChannel.currentSnapshot
            }
        }

        return null
    }

    /**
     * Remove a snapshot from the App.
     *
     * @param snapshotId the ID of a snapshot to clear from the app
     */
    fun removeSnapshot(snapshotId: String) {
        for (channel in channels) {
            if (channel.currentSnapshot == snapshotId) {
                channel.currentSnapshot = ""
            }
        }

        snapshots.remove(snapshotId)
    }
}