package io.ionic.liveupdates.data

import android.content.Context
import android.content.SharedPreferences
import io.ionic.liveupdates.LiveUpdate
import io.ionic.liveupdates.data.model.*
import io.ionic.liveupdates.data.model.network.request.CheckRequest
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import java.util.*

internal object DataManager {
    private const val PREFS_MAIN = "io.ionic.deploy.common"
    private const val PREFS_APPS = "io.ionic.deploy.apps"
    private const val PREFS_SNAPSHOTS = "io.ionic.deploy.versions"

    private val json = Json { explicitNulls = false; encodeDefaults = true; isLenient = true }

    /**
     * Creates and stores a UUID to identify this app with.
     */
    fun initialize(context: Context) {
        val sharedPrefs: SharedPreferences =
            context.getSharedPreferences(PREFS_MAIN, Context.MODE_PRIVATE)
        val deviceUUID = sharedPrefs.getString("uuid", UUID.randomUUID().toString())
        with(sharedPrefs.edit()) {
            putString("uuid", deviceUUID)
            apply()
        }
    }

    fun reset(context: Context, retainCache: Boolean = false) {
        with(context.getSharedPreferences(PREFS_APPS, Context.MODE_PRIVATE).edit()) {
            clear()
            apply()
        }

        if (!retainCache) {
            with(context.getSharedPreferences(PREFS_SNAPSHOTS, Context.MODE_PRIVATE).edit()) {
                clear()
                apply()
            }
        }
    }

    fun saveApp(context: Context, app: App) {
        val sharedPrefs: SharedPreferences =
            context.getSharedPreferences(PREFS_APPS, Context.MODE_PRIVATE)
        with(sharedPrefs.edit()) {
            putString(app.id, json.encodeToString(app))
            apply()
        }
    }

    fun addSnapshot(context: Context, snapshot: Snapshot) {
        val sharedPrefs: SharedPreferences =
            context.getSharedPreferences(PREFS_SNAPSHOTS, Context.MODE_PRIVATE)
        with(sharedPrefs.edit()) {
            putString(snapshot.id, json.encodeToString(snapshot))
            apply()
        }
    }

    fun deleteSnapshot(context: Context, snapshotId: String) {
        val sharedPrefs: SharedPreferences =
            context.getSharedPreferences(PREFS_SNAPSHOTS, Context.MODE_PRIVATE)
        with(sharedPrefs.edit()) {
            remove(snapshotId)
            apply()
        }
    }

    /**
     * Get app with the provided id
     */
    fun getApp(context: Context, appId: String): App? {
        val sharedPrefs: SharedPreferences =
            context.getSharedPreferences(PREFS_APPS, Context.MODE_PRIVATE)
        val appString = sharedPrefs.getString(appId, null)
        return if (appString != null) {
            json.decodeFromString(appString)
        } else {
            null
        }
    }

    fun getApps(context: Context): List<App> {
        val apps = mutableListOf<App>()
        val sharedPrefs: SharedPreferences =
            context.getSharedPreferences(PREFS_APPS, Context.MODE_PRIVATE)
        sharedPrefs.all.forEach { entry ->
            val app = json.decodeFromString<App>(entry.value as String)
            apps.add(app)
        }

        return apps
    }

    fun getSnapshots(context: Context): List<Snapshot> {
        val snapshots = mutableListOf<Snapshot>()
        val sharedPrefs: SharedPreferences =
            context.getSharedPreferences(PREFS_SNAPSHOTS, Context.MODE_PRIVATE)

        sharedPrefs.all.forEach { entry ->
            val snapshot = json.decodeFromString<Snapshot>(entry.value as String)
            snapshots.add(snapshot)
        }

        return snapshots
    }

    fun getSnapshots(context: Context, appId: String): List<Snapshot> {
        val snapshots = mutableListOf<Snapshot>()

        val app = getApp(context, appId)
        if (app != null) {
            for (snapshotId in app.snapshots) {
                val snapshot = getSnapshot(context, snapshotId)
                if (snapshot != null) {
                    snapshots.add(snapshot)
                }
            }
        }

        return snapshots
    }

    /**
     * Get a snapshot with the provided id
     */
    fun getSnapshot(context: Context, snapshotId: String): Snapshot? {
        val sharedPrefs: SharedPreferences =
            context.getSharedPreferences(PREFS_SNAPSHOTS, Context.MODE_PRIVATE)
        val snapshotString = sharedPrefs.getString(snapshotId, null)
        return if (snapshotString != null) {
            json.decodeFromString(snapshotString)
        } else {
            null
        }
    }

    /**
     * Get current snapshot with provided channel
     */
    fun getCurrentSnapshot(context: Context, channel: Channel): Snapshot? {
        val currentSnapshot = channel.currentSnapshot
        if (currentSnapshot.isNotEmpty()) {
            return getSnapshot(context, currentSnapshot)
        }

        return null
    }

    fun updateLastSync(context: Context, appId: String) {
        val app = getApp(context, appId)
        if (app != null) {
            app.lastSync = System.currentTimeMillis()
            saveApp(context, app)
        }
    }

    fun getLastSync(context: Context, appId: String): Long {
        val app = getApp(context, appId)
        if (app != null) {
            return app.lastSync
        }

        return -1
    }

    fun getCheckUpdateData(context: Context, instance: LiveUpdate): CheckRequest? {
        // Get app binary version from project
        val manager = context.packageManager
        val info = manager.getPackageInfo(context.packageName, 0)
        val binaryVersion = info.versionName

        // Get generated device UUID
        val sharedPrefs = context.getSharedPreferences(PREFS_MAIN, Context.MODE_PRIVATE)
        val deviceUUID = sharedPrefs.getString("uuid", "")!!

        // Device details
        val checkRequestDevice = Device(binaryVersion, deviceUUID)

        // Get app info
        val app = getApp(context, instance.appId) ?: return null

        // Check if Portals app
        var isPortalsPresent = false
        try {
            Class.forName("io.ionic.portals.PortalManager")
            isPortalsPresent = true
        } catch (e: ClassNotFoundException) {
            // Do nothing, isPortalsPresent remains false.
        }

        val checkRequest = CheckRequest(checkRequestDevice, instance.appId, instance.channelName, isPortalsPresent)
        val currentSnapshot = app.getCurrentSnapshot(instance.channelName)
        if (currentSnapshot != null) {
            val snapshot = getSnapshot(context, currentSnapshot)
            if (snapshot != null) {
                checkRequestDevice.snapshot = snapshot.id
                checkRequestDevice.build = snapshot.buildId
            }
        }

        return checkRequest
    }
}