package io.ionic.liveupdates.network

import android.content.Context
import android.util.Log
import io.ionic.liveupdates.LiveUpdateManager
import io.ionic.liveupdates.data.model.network.request.CheckRequest
import io.ionic.liveupdates.data.model.network.response.DownloadResponse
import io.ionic.liveupdates.data.model.network.response.CheckResponse
import io.ionic.liveupdates.data.model.network.response.ErrorResponse
import io.ionic.liveupdates.data.model.network.response.SuccessResponse
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import okhttp3.*
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.RequestBody.Companion.toRequestBody
import okio.IOException
import java.io.File
import java.io.FileOutputStream

object Client {
    private const val API_PROD = "https://api.ionicjs.com"
    private const val API_STAGE = "https://api-staging.ionicjs.com"

    private val client = OkHttpClient()

    private val json = Json { explicitNulls = false; encodeDefaults = true; isLenient = true }

    private fun getEndpointCheck(appId: String): String {
        return "${API_PROD}/apps/${appId}/channels/check-device"
    }

    private fun getEndpointDownload(appId: String, snapshotID: String): String {
        return "${API_PROD}/apps/${appId}/snapshots/${snapshotID}/download"
    }

    fun checkForUpdate(checkRequest: CheckRequest): CheckResponse? {
        val requestBody = json.encodeToString(checkRequest)
            .toRequestBody("application/json; charset=utf-8".toMediaType())
        val request = Request.Builder()
            .url(getEndpointCheck(checkRequest.app_id))
            .post(requestBody)
            .build()

        client.newCall(request).execute().use { response ->
            if (!response.isSuccessful) {
                if (response.body != null) {
                    val bodyString = response.body!!.string()
                    val errorResponse = json.decodeFromString<ErrorResponse>(bodyString)
                    Log.e(
                        LiveUpdateManager.TAG,
                        "Check update request failed for " + checkRequest.app_id
                    )
                    Log.e(LiveUpdateManager.TAG, bodyString)
                    return CheckResponse(null, errorResponse)
                }
            }

            if (response.body != null) {
                val bodyString = response.body!!.string()
                val successResponse = json.decodeFromString<SuccessResponse>(bodyString)
                return CheckResponse(successResponse, null)
            }

            return null
        }
    }

    fun downloadUpdate(context: Context, appId: String, snapshotId: String): DownloadResponse? {
        val appDir = File(LiveUpdateManager.getLiveUpdatesDirectory(context), appId)

        val request = Request.Builder()
            .url(getEndpointDownload(appId, snapshotId))
            .build()

        client.newCall(request).execute().use { response ->
            if (!response.isSuccessful) {
                if (response.body != null) {
                    val bodyString = response.body!!.string()
                    val errorResponse = json.decodeFromString<ErrorResponse>(bodyString)
                    Log.e(
                        LiveUpdateManager.TAG,
                        "Download update request failed for app $appId snapshot $snapshotId"
                    )
                    Log.e(LiveUpdateManager.TAG, bodyString)
                    return DownloadResponse(errorResponse, null)
                }

                return null
            }

            try {
                val fileData = response.body?.byteStream()
                if (fileData != null) {
                    val fileOutputStream = FileOutputStream(File(appDir, "${snapshotId}.zip"))
                    fileOutputStream.write(fileData.readBytes())
                    fileOutputStream.close()
                    fileData.close()
                    Log.d(
                        LiveUpdateManager.TAG,
                        "Snapshot ${snapshotId}.zip downloaded for app $appId"
                    )
                }
            } catch (exception: IOException) {
                Log.e(
                    LiveUpdateManager.TAG,
                    "Saving download failed for app $appId snapshot $snapshotId"
                )
                Log.e(LiveUpdateManager.TAG, exception.stackTraceToString())
                return null
            }

            return DownloadResponse(null, File(appDir, "${snapshotId}.zip"))
        }
    }
}