package io.ionic.liveupdates

import android.content.Context
import android.util.Log
import io.ionic.liveupdates.data.DataManager
import io.ionic.liveupdates.data.model.App
import io.ionic.liveupdates.data.model.Channel
import io.ionic.liveupdates.data.model.Snapshot
import io.ionic.liveupdates.data.model.network.response.DownloadResponse
import io.ionic.liveupdates.data.model.network.response.CheckResponse
import io.ionic.liveupdates.network.*
import kotlinx.coroutines.*
import kotlinx.coroutines.sync.Semaphore
import java.io.File
import java.io.IOException
import java.util.concurrent.ConcurrentHashMap
import kotlin.coroutines.coroutineContext

/**
 * The Live Update Manager provides the ability to download and cache updates to web applications
 * used in the Ionic ecosystem.
 */
object LiveUpdateManager {

    /**
     * Log tag string.
     */
    internal const val TAG = "LiveUpdates"

    /**
     * Parent directory used by live updates to store web app files.
     */
    private const val LIVE_UPDATES_DIR = "ionic_apps"

    /**
     * Apps to maintain using Live Updates.
     */
    private val instances: MutableMap<String, LiveUpdate> = mutableMapOf()

    /**
     * The maximum number of updates per app instance to be stored locally on the device.
     */
    var maxVersions: Int = 3

    /**
     * General coroutine scope.
     */
    private val liveUpdateScope = CoroutineScope(Dispatchers.Default)

    /**
     * Coroutine scope used for network and IO operations.
     */
    private val liveUpdateIOScope = CoroutineScope(Dispatchers.IO)

    /**
     * Coroutine scope used for UI operations.
     */
    private val mainScope = CoroutineScope(Dispatchers.Main)

    /**
     * Track any running sync jobs
     */
    private val syncJobs: ConcurrentHashMap<String, Job> = ConcurrentHashMap()

    /**
     * Semaphore used to limit the sync operation to once per app at a time
     */
    private val syncSemaphore = Semaphore(1)

    /**
     * Initializes the file directory and shared preferences used to save update data.
     * @param context context used to initialize
     */
    @JvmStatic
    fun initialize(context: Context) {
        setupMainDirectory(context)
        DataManager.initialize(context)
    }

    /**
     * Clears the live updates directory and saved update data.
     * Attempts to cancel any running sync jobs.
     * @param context context used to reset
     */
    @JvmStatic
    @JvmOverloads
    fun reset(context: Context, retainCache: Boolean = false) {
        // Attempt to cancel any running sync jobs
        cancelSync()

        // Recreate main and app directories
        setupMainDirectory(context, reset = !retainCache)
        instances.map {
            setupAppDirectory(context, it.key)
            it.value.appState = AppState.UNCHECKED
        }

        // Recreate shared preferences
        DataManager.reset(context, retainCache)
        for (instance in instances) {
            storeAppData(context, instance.value)
        }

        Log.d(TAG, "Live Updates data has been reset!")
    }

    /**
     * Adds an instance of a Live Update.
     * @param context context used to store app data
     * @param liveUpdate The LiveUpdate to add
     */
    @JvmStatic
    fun addLiveUpdateInstance(context: Context, liveUpdate: LiveUpdate) {
        instances[liveUpdate.appId] = liveUpdate
        storeAppData(context, liveUpdate)
        setupAppDirectory(context, liveUpdate.appId)
    }

    /**
     * Check when an app was last synced.
     * @return the timestamp of the last sync in milliseconds, or -1 if no prior sync
     */
    @JvmStatic
    fun getLastSync(context: Context, appId: String): Long {
        return DataManager.getLastSync(context, appId)
    }

    /**
     * Check the latest time any of the registered apps was synced.
     * @return the timestamp of the oldest synced registered app, or -1 if any app has never
     * been synced
     */
    @JvmStatic
    fun getLastSync(context: Context): Long {
        var oldestSync = Long.MAX_VALUE
        instances.map { instance ->
            val app = DataManager.getApp(context, instance.value.appId)!!
            val appSyncTime = app.lastSync
            if (appSyncTime < oldestSync) {
                oldestSync = appSyncTime
            }
        }

        return if (oldestSync == Long.MAX_VALUE) {
            -1
        } else {
            oldestSync
        }
    }

    /**
     * Checks a single app with live updates to see if any new builds are available. If they are,
     * attempt to download, unzip, and update the saved data to reference the latest build.
     * @param context context used during app update operation
     * @param appId an appId to sync
     * @param callback a callback to notify on each app sync complete and when sync is complete
     */
    @JvmStatic
    @JvmOverloads
    fun sync(context: Context, appId: String, callback: SyncCallback? = null) {
        sync(context, arrayOf(appId), callback = callback)
    }

    /**
     * Checks all apps or a provided set of apps with live updates to see if any new builds are
     * available. If they are, attempt to download, unzip, and update the saved data to reference
     * the latest build.
     * @param context context used during app update operation
     * @param appIds an array of appId's to sync, default will sync all apps
     * @param async whether to run updates in parallel or not
     * @param callback a callback to notify on each app sync complete and when sync is complete
     */
    @JvmStatic
    @JvmOverloads
    fun sync(
        context: Context,
        appIds: Array<String> = arrayOf(),
        async: Boolean = true,
        callback: SyncCallback? = null
    ) {
        val syncApps: Map<String, LiveUpdate> = if (appIds.isEmpty()) {
            // Sync all apps
            instances
        } else {
            // Sync only provided appIds
            appArrayToInstances(appIds)
        }

        // syncApps could be empty if no registered apps matched the provided appIds
        if (syncApps.isEmpty()) {
            callback?.onSyncComplete()
            return
        }

        liveUpdateScope.launch {
            if (async) {
                // Async mode will sync all provided appIds in parallel
                withContext(liveUpdateScope.coroutineContext) {
                    syncApps.map {
                        val appId = it.key
                        syncSemaphore.acquire()
                        if (!syncJobs.containsKey(appId) || (syncJobs.containsKey(appId) && !syncJobs[appId]!!.isActive)) {
                            liveUpdateIOScope.launch {
                                val job = this.coroutineContext.job
                                syncJobs[appId] = job
                                syncSemaphore.release()

                                try {
                                    syncApp(context, appId, callback)
                                } catch (e: CancellationException) {
                                    Log.d(TAG, "Update task canceled for $appId")
                                    val liveUpdateInstance = instances[appId]
                                    liveUpdateInstance?.appState = AppState.CANCELED

                                    mainScope.launch {
                                        callback?.onAppComplete(it.value, FailStep.CANCEL)
                                    }
                                }

                                syncSemaphore.acquire()
                                syncJobs.remove(appId)
                                syncSemaphore.release()
                            }
                        } else {
                            Log.d(TAG, "Skipped adding sync task for $appId, one in progress!")
                            syncSemaphore.release()

                            mainScope.launch {
                                callback?.onAppComplete(it.value, FailStep.CHECK)
                            }
                        }
                    }
                }.joinAll()
            } else {
                // Non-Async mode will sync all provided appIds sequentially
                syncApps.map {
                    val appId = it.key
                    syncSemaphore.acquire()
                    if (!syncJobs.containsKey(appId) || (syncJobs.containsKey(appId) && !syncJobs[appId]!!.isActive)) {
                        try {
                            withContext(liveUpdateIOScope.coroutineContext) {
                                val job = this.coroutineContext.job
                                syncJobs[appId] = job
                                syncSemaphore.release()

                                syncApp(context, appId, callback)
                            }
                        } catch (e: CancellationException) {
                            Log.d(TAG, "Update task canceled for $appId")
                            val liveUpdateInstance = instances[appId]
                            liveUpdateInstance?.appState = AppState.CANCELED
                            mainScope.launch {
                                callback?.onAppComplete(it.value, FailStep.CANCEL)
                            }

                            // If we want to cancel the full sync, break loop here.
                        }

                        syncSemaphore.acquire()
                        syncJobs.remove(appId)
                        syncSemaphore.release()
                    } else {
                        Log.d(TAG, "Skipped adding sync task for $appId, one in progress!")
                        syncSemaphore.release()
                        mainScope.launch {
                            callback?.onAppComplete(it.value, FailStep.CHECK)
                        }
                    }
                }
            }

            mainScope.launch {
                callback?.onSyncComplete()
            }
        }
    }

    private suspend fun syncApp(context: Context, appId: String, callback: SyncCallback? = null) {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Log.e(TAG, "App ID $appId not setup for live updates!")
            return
        }

        // Check for updates
        val checkResponse = checkForUpdates(context, liveUpdateInstance)
        if (checkResponse == null || checkResponse.error != null) {
            mainScope.launch {
                callback?.onAppComplete(liveUpdateInstance, FailStep.CANCEL)
            }
            return
        }

        coroutineContext.ensureActive()

        // Download available update
        val bodyData = checkResponse.success!!.data
        val available = bodyData.available
        if (available) {
            val snapshotId = checkResponse.success.data.snapshot
            val buildId = checkResponse.success.data.build

            if (snapshotId == null || buildId == null) {
                mainScope.launch {
                    callback?.onAppComplete(liveUpdateInstance, FailStep.CHECK)
                }
                return
            }

            // Check cache before downloading
            Log.d(TAG, "Update available for app $appId. Checking cache for ${snapshotId}...")
            val appSnapshots = getSnapshots(context)
            val snapshotExists = appSnapshots.filter { it.id == snapshotId }
            if (snapshotExists.isEmpty() || !getSnapshotDirectory(
                    context,
                    appId,
                    snapshotId
                ).exists()
            ) {
                // Download
                Log.d(
                    TAG,
                    "Snapshot not present in cache for $appId. Downloading snapshot ${snapshotId}..."
                )
                val zipFile = download(context, liveUpdateInstance, snapshotId)
                if (zipFile == null) {
                    mainScope.launch {
                        callback?.onAppComplete(liveUpdateInstance, FailStep.DOWNLOAD)
                    }
                    return
                }

                coroutineContext.ensureActive()

                // Unzip
                val prepared = prepareSnapshot(liveUpdateInstance, zipFile)
                if (prepared == null) {
                    mainScope.launch {
                        callback?.onAppComplete(liveUpdateInstance, FailStep.UNPACK)
                    }
                    return
                }

                // Cleanup the zip file
                zipFile.delete()
            } else {
                Log.d(TAG, "Cache for app $appId contains snapshot ${snapshotId}, reapplying...")
            }

            // Update
            val updateSuccessful = applyUpdate(context, liveUpdateInstance, snapshotId, buildId)
            if (!updateSuccessful) {
                mainScope.launch {
                    callback?.onAppComplete(liveUpdateInstance, FailStep.UPDATE)
                }
                return
            }

            DataManager.updateLastSync(context, appId)

            val appDir = getLatestAppDirectory(context, appId)
            if (appDir != null) {
                Log.d(TAG, "App $appId path updated to: ${appDir.path}")
            }

            mainScope.launch {
                callback?.onAppComplete(liveUpdateInstance, null)
            }
        } else {
            DataManager.updateLastSync(context, appId)
            liveUpdateInstance.appState = AppState.CHECKED
            liveUpdateInstance.availableUpdate = null
            Log.d(TAG, "No update available for app $appId.")
        }
    }

    /**
     * Cancels a running sync job
     * @param appId the ID of the app associated with a sync job to cancel
     */
    @JvmStatic
    fun cancelSync(appId: String) {
        val job = syncJobs[appId]
        job?.cancel()
    }

    /**
     * Cancel all running sync jobs
     */
    @JvmStatic
    fun cancelSync() {
        syncJobs.map { syncJob ->
            syncJob.value.cancel()
            Log.d(TAG, "Canceling job ${syncJob.key}")
        }
    }

    /**
     * Clean up any stale app versions.
     * Stale versions are any app snapshot files built for previous versions of the app binary and
     * not currently used by any app channel.
     * @param context the context to use
     * @param appId the ID of the app to clean
     * @param callback a callback to handle the response to the cleanup once completed
     */
    @JvmStatic
    fun cleanStaleVersions(context: Context, appId: String) {
        // Get app binary version
        val manager = context.packageManager
        val info = manager.getPackageInfo(context.packageName, 0)
        val binaryVersion = info.versionName

        val app = DataManager.getApp(context, appId)
        if (app != null) {
            // Keep a cache of any snapshot that is currently used by a channel in case it needs
            // to be used again.
            val snapshotsInUse = mutableListOf<String>()

            val channels = app.channels
            channels.map { channel ->
                val channelSnapshot = channel.currentSnapshot
                snapshotsInUse.add(channelSnapshot)

                val snapshotObj = DataManager.getSnapshot(context, channel.currentSnapshot)
                if (snapshotObj != null && snapshotObj.binaryVersion != binaryVersion) {
                    Log.d(
                        TAG,
                        "Update ${channel.currentSnapshot} for app ${app.id} was built for a different binary version (v${snapshotObj.binaryVersion}) caching..."
                    )
                    channel.currentSnapshot = ""
                }
            }

            val snapshotsToRemove = mutableListOf<String>()
            val appSnapshots = app.snapshots.toMutableList()
            appSnapshots.removeAll(snapshotsInUse)
            appSnapshots.map { snapshotId ->
                val snapshotObj = DataManager.getSnapshot(context, snapshotId)
                if (snapshotObj != null && snapshotObj.binaryVersion != binaryVersion) {
                    Log.d(
                        TAG,
                        "Update $snapshotId for app ${app.id} was built for a different binary version (v${snapshotObj.binaryVersion}) and no longer used, removing from device"
                    )
                    snapshotsToRemove.add(snapshotId)
                    removeSnapshot(context, app.id, snapshotId)
                }
            }

            app.snapshots.removeAll(snapshotsToRemove)

            DataManager.saveApp(context, app)
        }
    }

    /**
     * Clean up unused/old app versions.
     * @param context the context to use
     */
    @JvmStatic
    fun cleanVersions(context: Context) {
        instances.map { app ->
            cleanVersions(context, app.key)
        }
    }

    /**
     * Clean up unused/old app versions related to a specific app.
     * @param context the context to use
     * @param appId the ID of the app to clean
     */
    @JvmStatic
    fun cleanVersions(context: Context, appId: String) {
        // Clean stale versions first
        cleanStaleVersions(context, appId)

        val app = DataManager.getApp(context, appId)
        if (app != null) {
            val snapshotsInUse = mutableListOf<String>()
            val snapshotsToRemove = mutableListOf<String>()

            // Get snapshots in use we don't want to remove
            app.channels.map { channel ->
                snapshotsInUse.add(channel.currentSnapshot)
            }

            // Order remaining snapshots by last used date and remove any over the maxVersions cap
            val remainingSnapshots = app.snapshots.toMutableList()
            remainingSnapshots.removeAll(snapshotsInUse)
            remainingSnapshots.sortByDescending {
                val snapshotObj = DataManager.getSnapshot(context, it)
                snapshotObj!!.lastUsed
            }

            val maxIndex = maxOf(remainingSnapshots.size, maxVersions)
            val trimSnapshots = remainingSnapshots.slice(maxVersions until maxIndex)
            trimSnapshots.map { snapshot ->
                Log.d(TAG, "Cleanup - Removing excess app $appId snapshot $snapshot")
                snapshotsToRemove.add(snapshot)
                removeSnapshot(context, appId, snapshot)
            }

            app.snapshots.removeAll(snapshotsToRemove)
            DataManager.saveApp(context, app)
        }
    }

    /**
     * Synchronous function to check for an update for an app.
     * @param context the context to use
     * @param appId the ID of the app to check
     * @return the response
     */
    @JvmStatic
    fun checkForUpdate(context: Context, appId: String): CheckResponse? {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Log.e(TAG, "App ID $appId not setup for live updates!")
            return null
        }

        return checkForUpdates(context, liveUpdateInstance)
    }

    /**
     * Asynchronous function to check for an update for an app.
     * @param context the context to use
     * @param appId the ID of the app to check
     * @param callback a callback to handle the response from the update check request
     */
    @JvmStatic
    fun checkForUpdate(context: Context, appId: String, callback: CheckCallback) {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Log.e(TAG, "App ID $appId not setup for live updates!")
            callback.onComplete(null)
            return
        }

        liveUpdateIOScope.launch {
            val response = checkForUpdates(context, liveUpdateInstance)
            callback.onComplete(response)
        }
    }

    /**
     * Synchronous function to download an update for an app.
     * @param context the context to use
     * @param appId the ID of the app
     * @param snapshotId the snapshot id to download
     * @return the response
     */
    @JvmStatic
    fun downloadUpdate(context: Context, appId: String, snapshotId: String): DownloadResponse? {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Log.e(TAG, "App ID $appId not setup for live updates!")
            return null
        }

        return Client.downloadUpdate(context, appId, snapshotId)
    }

    /**
     * Asynchronous function to check for an update for an app.
     * @param context the context to use
     * @param appId the ID of the app
     * @param snapshotId the snapshot id to download
     * @param callback a callback to handle the response from the download request
     */
    @JvmStatic
    fun downloadUpdate(
        context: Context,
        appId: String,
        snapshotId: String,
        callback: DownloadCallback
    ) {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Log.e(TAG, "App ID $appId not setup for live updates!")
            callback.onComplete(null)
            return
        }

        liveUpdateIOScope.launch {
            val response = Client.downloadUpdate(context, appId, snapshotId)
            callback.onComplete(response)
        }
    }

    /**
     * Synchronous function to unzip a downloaded app.
     * @param appId the ID of the app
     * @param zipFile the zipped app download to extract
     * @return a file directory where the app was unzipped
     */
    @JvmStatic
    fun extractUpdate(appId: String, zipFile: File): File? {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Log.e(TAG, "App ID $appId not setup for live updates!")
            return null
        }

        return prepareSnapshot(liveUpdateInstance, zipFile)
    }

    /**
     * Asynchronous function to unzip a downloaded app.
     * @param zipFile the zipped app download to extract
     * @param callback a callback to handle the response from the downlaod request
     */
    @JvmStatic
    fun extractUpdate(appId: String, zipFile: File, callback: ExtractCallback) {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Log.e(TAG, "App ID $appId not setup for live updates!")
            callback.onComplete(null)
            return
        }

        liveUpdateIOScope.launch {
            val file = prepareSnapshot(liveUpdateInstance, zipFile)
            callback.onComplete(file)
        }
    }

    /**
     * Apply a downloaded and extracted Live Update snapshot to the portal for next load.
     * @param context the context to use
     * @param appId the ID of the app to apply a snapshot update to
     * @param snapshotId the ID of the snapshot to apply
     * @param buildId the ID of the build to apply
     */
    @JvmStatic
    fun applyUpdate(context: Context, appId: String, snapshotId: String, buildId: String) {
        val liveUpdateInstance = instances[appId]
        if (liveUpdateInstance == null) {
            Log.e(TAG, "App ID $appId not setup for live updates!")
            return
        }

        applyUpdate(context, liveUpdateInstance, snapshotId, buildId)
    }

    /**
     * Get the map of all registered apps using Live Updates.
     * @return a map of all registered apps that use Live Updates.
     */
    @JvmStatic
    fun getApps(): MutableMap<String, LiveUpdate> {
        return instances
    }

    /**
     * Get the latest directory for the updated app resources, if available. If the app has
     * not been updated through Live Updates, null will be returned.
     * @param context context used to get the latest directory
     * @param appId the ID of the app directory to retrieve
     * @return directory file if available, otherwise null
     */
    @JvmStatic
    fun getLatestAppDirectory(context: Context, appId: String): File? {
        val instance = instances[appId]
        if (instance != null) {
            val app = DataManager.getApp(context, appId)
            if (app != null) {
                val channel = app.getChannel(instance.channelName)
                if (channel != null) {
                    if (channel.currentSnapshot.isNotEmpty()) {
                        return getSnapshotDirectory(context, appId, channel.currentSnapshot)
                    }
                }
            }
        }

        return null
    }

    internal fun getSnapshots(context: Context): List<Snapshot> {
        return DataManager.getSnapshots(context)
    }

    internal fun getLiveUpdatesDirectory(context: Context): File {
        return File(context.filesDir, LIVE_UPDATES_DIR)
    }

    private fun getSnapshotDirectory(context: Context, appId: String, snapshotId: String): File {
        val filePath =
            context.filesDir.path + File.separator + LIVE_UPDATES_DIR + File.separator + appId + File.separator + snapshotId
        return File(filePath)
    }

    private fun appArrayToInstances(appIds: Array<String>): MutableMap<String, LiveUpdate> {
        val map: MutableMap<String, LiveUpdate> = mutableMapOf()
        appIds.map {
            if (instances.containsKey(it)) {
                map[it] = instances[it]!!
            }
        }

        return map
    }

    private fun checkForUpdates(context: Context, instance: LiveUpdate): CheckResponse? {
        instance.appState = AppState.CHECKING
        instance.availableUpdate = null

        val checkRequest = DataManager.getCheckUpdateData(context, instance) ?: return null

        val responseBody = Client.checkForUpdate(checkRequest)
        if (responseBody != null) {
            if (responseBody.error != null) {
                Log.e(TAG, responseBody.error.toString())
                instance.appState = AppState.FAILED
                instance.availableUpdate = null
            } else {
                val bodyData = responseBody.success!!.data
                val available = bodyData.available
                if (available) {
                    instance.appState = AppState.AVAILABLE
                    instance.availableUpdate = AvailableUpdateState.AVAILABLE
                }
            }
        }

        return responseBody
    }

    private fun download(context: Context, instance: LiveUpdate, snapshotId: String): File? {
        instance.appState = AppState.DOWNLOADING

        val download = Client.downloadUpdate(context, instance.appId, snapshotId)
        if (download?.error != null) {
            instance.appState = AppState.FAILED
            Log.e(TAG, "Error downloading update for app ${instance.appId} snapshot $snapshotId.")
            return null
        }

        download?.file.let {
            Log.d(TAG, "The saved ${it!!.path} file is ${it.length()} bytes")
            instance.appState = AppState.DOWNLOADED
            instance.availableUpdate = AvailableUpdateState.PENDING
            return it
        }
    }

    private fun setupMainDirectory(context: Context, reset: Boolean = false) {
        val liveUpdateDir = getLiveUpdatesDirectory(context)

        // if resetting, delete the whole dir and start fresh
        if (reset) {
            liveUpdateDir.deleteRecursively()
        }

        // create live updates directory if it doesn't exist
        if (!liveUpdateDir.exists()) {
            liveUpdateDir.mkdir()
        }
    }

    private fun setupAppDirectory(context: Context, appId: String) {
        val appDir = File(getLiveUpdatesDirectory(context), appId)

        // create app directory if it doesn't exist
        if (!appDir.exists()) {
            appDir.mkdir()
        }
    }

    private fun prepareSnapshot(instance: LiveUpdate, file: File): File? {
        instance.appState = AppState.UNPACKING

        return try {
            val resultDir = ZipUtility.unzip(file)
            instance.appState = AppState.UNPACKED
            resultDir
        } catch (exception: IOException) {
            instance.appState = AppState.FAILED
            Log.e(
                TAG,
                "Error preparing downloaded update for app ${instance.appId} file ${file.path}."
            )
            Log.e(TAG, "Error unzipping ${file.path}!")
            Log.e(TAG, exception.stackTraceToString())
            null
        }
    }

    private fun applyUpdate(
        context: Context,
        instance: LiveUpdate,
        snapshotId: String,
        buildId: String
    ): Boolean {
        instance.appState = AppState.UPDATING

        // Get app binary version
        val manager = context.packageManager
        val info = manager.getPackageInfo(context.packageName, 0)

        var currentSnapshot = DataManager.getSnapshot(context, snapshotId)
        if (currentSnapshot == null) {
            currentSnapshot =
                Snapshot(snapshotId, buildId, info.versionName, System.currentTimeMillis())
        } else {
            currentSnapshot.lastUsed = System.currentTimeMillis()
            currentSnapshot.binaryVersion = info.versionName
        }

        DataManager.addSnapshot(context, currentSnapshot)

        val app = DataManager.getApp(context, instance.appId)
        if (app != null) {
            val currentChannel =
                app.channels.filter { channel -> channel.id == instance.channelName }
            if (currentChannel.isNotEmpty()) {
                val channel = currentChannel[0]
                channel.currentSnapshot = snapshotId
                if (!app.snapshots.contains(snapshotId)) {
                    app.snapshots.add(snapshotId)
                }

                DataManager.saveApp(context, app)

                instance.appState = AppState.UPDATED
                instance.availableUpdate = AvailableUpdateState.READY
                return true
            } else {
                instance.appState = AppState.FAILED
                Log.e(TAG, "Error retrieving App channel data from local store.")
            }
        } else {
            instance.appState = AppState.FAILED
            Log.e(TAG, "Error retrieving App data from local store.")
        }

        return false
    }

    private fun removeSnapshot(context: Context, appId: String, snapshotId: String) {
        // remove from shared prefs
        DataManager.deleteSnapshot(context, snapshotId)

        // delete files
        val snapshotDir = getSnapshotDirectory(context, appId, snapshotId)
        if (snapshotDir.exists()) {
            snapshotDir.deleteRecursively()
        }
    }

    private fun storeAppData(context: Context, liveUpdate: LiveUpdate) {
        var app = DataManager.getApp(context, liveUpdate.appId)
        if (app == null) {
            app = App(liveUpdate.appId)
        }

        app.addChannel(Channel(liveUpdate.channelName))
        DataManager.saveApp(context, app)
    }
}