package io.iohk.atala.prism.credentials.json

import io.iohk.atala.prism.credentials.CredentialParsingError
import io.iohk.atala.prism.credentials.PrismCredential
import io.iohk.atala.prism.credentials.content.CredentialContent
import io.iohk.atala.prism.crypto.EC
import io.iohk.atala.prism.crypto.keys.ECPrivateKey
import io.iohk.atala.prism.crypto.signature.ECSignature
import io.iohk.atala.prism.protos.util.Base64Utils
import kotlin.js.JsExport
import kotlin.jvm.JvmStatic

/**
 * @see [PrismCredential]
 *
 * Implementation of [PrismCredential] based on the json representation [CredentialContent].
 *
 * @property content is the json representation of a prism credential
 * @property signature is the signature this credential was signed with.
 */
@JsExport
public data class JsonBasedCredential constructor(
    override val content: CredentialContent,
    override val signature: ECSignature? = null
) : PrismCredential() {

    /**
     * @see PrismCredential.contentBytes
     */
    override val contentBytes: ByteArray = content.fields.toString().encodeToByteArray()

    /**
     * @see PrismCredential.canonicalForm
     */
    override val canonicalForm: String =
        if (signature != null) {
            Base64Utils.encode(contentBytes) +
                SEPARATOR +
                Base64Utils.encode(signature.getEncoded())
        } else {
            contentBytes.decodeToString()
        }

    /**
     * @see PrismCredential.sign
     */
    override fun sign(privateKey: ECPrivateKey): PrismCredential {
        return copy(signature = EC.signBytes(contentBytes, privateKey))
    }

    public companion object {
        private val SEPARATOR = '.'

        /**
         * @return a new [JsonBasedCredential] parsed from the serialised version of an unsigned @param[credential].
         * @param[credential] must be a base64Url string
         */
        @JvmStatic
        private fun parseUnsignedCredential(credential: String): JsonBasedCredential {
            val credentialContent = CredentialContent.fromString(Base64Utils.decode(credential).decodeToString())
            return JsonBasedCredential(credentialContent)
        }

        /**
         * @return a new [JsonBasedCredential] parsed from the serialised version of a signed @param[credential].
         * @param[credential] must be 2 Base64Url strings concatenated by `.` character, credential json on the left and signature on the right
         */
        @JvmStatic
        private fun parseSignedCredential(credential: String): JsonBasedCredential {
            val contentWithSignature = credential.split(SEPARATOR)
            if (contentWithSignature.size == 2) {
                val content = contentWithSignature[0]
                val signature = contentWithSignature[1]

                try {
                    val credentialContent =
                        CredentialContent.fromString(Base64Utils.decode(content).decodeToString())
                    return JsonBasedCredential(
                        content = credentialContent,
                        signature = ECSignature(Base64Utils.decode(signature))
                    )
                } catch (e: Exception) {
                    throw CredentialParsingError("Failed to parse signed credential content: ${e.message}")
                }
            } else {
                throw CredentialParsingError(
                    "Failed to parse signed credential. " +
                        "Expected format: [encoded credential]$SEPARATOR[encoded signature]"
                )
            }
        }

        /**
         * @return a new [JsonBasedCredential] parsed from the serialised version of either a signed or unsigned @param[credential].
         */
        @JvmStatic
        public fun fromString(credential: String): JsonBasedCredential {
            try {
                return parseUnsignedCredential(credential)
            } catch (e: Exception) {
                return parseSignedCredential(credential)
            }
        }
    }
}
