package io.iohk.atala.prism.credentials.content

import io.iohk.atala.prism.identity.PrismDid
import kotlinx.serialization.json.*
import kotlin.js.JsExport
import kotlin.jvm.JvmStatic

/**
 * Credential content JSON representation,
 * this class allows access to nested fields
 * with 'dot' notation like: credentialSubject.fieldName
 *
 * Example:
 * {{{
 *   import kotlinx.serialization.json.*
 *   val credentialContent = buildCredentialContent {
 *     put("issuerDid", "did")
 *     putJsonObject("credentialSubject") {
 *       put("fieldName", 123)
 *     }
 *   }
 *   credentialContent.getString("issuerDid") == "did"
 *   credentialContent.getInt("credentialSubject.fieldName") == 123
 * }}}
 *
 * This class wraps a @property[JsonObject].
 */
@JsExport
public data class CredentialContent(val fields: JsonObject) {

    /**
     * @return the content of the json @param[field] as [String] if it exists. Otherwise, @return[null].
     */
    public fun getString(field: String): String? =
        getField(field)?.jsonPrimitive?.contentOrNull

    /**
     * @return the content of the json @param[field] as [Int] if it exists. Otherwise, @return[null].
     */
    public fun getInt(field: String): Int? =
        getField(field)?.jsonPrimitive?.int

    /**
     * @return the content of the json @param[field] as [Boolean] if it exists. Otherwise, @return[null].
     */
    public fun getBoolean(field: String): Boolean? =
        getField(field)?.jsonPrimitive?.boolean

    /**
     * @return the content of the json @param[field] as [JsonArray] if it exists. Otherwise, @return[null].
     */
    public fun getArray(field: String): JsonArray? =
        getField(field)?.jsonArray

    /**
     * @return the content of the json @param[fieldName] as [JsonElement] if it exists. Otherwise, @return[null].
     */
    public fun getField(fieldName: String): JsonElement? {
        val path = fieldName.split('.')
        return path.drop(1).fold(fields[path[0]]) { field, key ->
            field?.jsonObject?.get(key)
        }
    }

    /**
     * @return the content of the json field 'id' as [PrismDid] if it exists. Otherwise, @return[null].
     */
    public fun getIssuerDid(): PrismDid? =
        getString(JsonField.IssuerDid.value)?.let {
            PrismDid.fromString(it)
        }

    /**
     * @return the content of the json field 'keyId' as [String] if it exists. Otherwise, @return[null].
     */
    public fun getIssuanceKeyId(): String? =
        getString(JsonField.IssuanceKeyId.value)

    /**
     * @return the content of the json field 'credentialSubject' as [String] if it exists. Otherwise, @return[null].
     */
    public fun getCredentialSubject(): JsonElement? =
        getField(JsonField.CredentialSubject.value)

    public companion object {
        /**
         * @return new [CredentialContent] parsed from the provided json @param[value].
         */
        @JvmStatic
        public fun fromString(value: String): CredentialContent =
            CredentialContent(Json.parseToJsonElement(value).jsonObject)
    }
}

internal enum class JsonField(public val value: String) {
    CredentialType("type"),
    Issuer("issuer"),
    IssuerDid("id"),
    IssuerName("name"),
    IssuanceKeyId("keyId"),
    IssuanceDate("issuanceDate"),
    ExpiryDate("expiryDate"),
    CredentialSubject("credentialSubject")
}
