package io.iohk.atala.prism.credentials

import io.iohk.atala.prism.credentials.content.CredentialContent
import io.iohk.atala.prism.crypto.EC
import io.iohk.atala.prism.crypto.Sha256
import io.iohk.atala.prism.crypto.Sha256Digest
import io.iohk.atala.prism.crypto.keys.ECPrivateKey
import io.iohk.atala.prism.crypto.keys.ECPublicKey
import io.iohk.atala.prism.crypto.signature.ECSignature
import kotlin.js.JsExport

/**
 * Abstract class that defines an api for viewing and interacting with a PRISM Credential along with its signature.
 */
@JsExport
public abstract class PrismCredential {
    /**
     * @return the byte representation of the Credential as [ByteArray].
     */
    public abstract val contentBytes: ByteArray

    /**
     * @return the json representation of the Credential as [CredentialContent].
     */
    public abstract val content: CredentialContent

    /**
     * @return the signature of the signee of the Credential as [ECSignature]. If credential is not signed, @return[null].
     */
    public abstract val signature: ECSignature?

    /**
     * @return the Canonical form of the Credential as [String].
     */
    public abstract val canonicalForm: String

    /**
     * @return True if credential is signed hence verifiable. Otherwise, False if credential is not signed.
     */
    public fun isVerifiable(): Boolean = signature != null

    /**
     * @return the hash representation as [Sha256Digest] of the Credential as [CredentialContent].
     */
    public fun hash(): Sha256Digest = Sha256.compute(canonicalForm.encodeToByteArray())

    /**
     * @return a signed copy of [PrismCredential] using provided @param[ECPrivateKey].
     */
    public abstract fun sign(privateKey: ECPrivateKey): PrismCredential

    /**
     * @return True if credential is signed by provided @param[ECPublicKey]. Otherwise, False.
     */
    public fun isValidSignature(publicKey: ECPublicKey): Boolean {
        return signature?.let { EC.verifyBytes(contentBytes, publicKey, it) } ?: false
    }
}
