package io.iohk.atala.prism.credentials

import com.benasher44.uuid.uuid4
import io.iohk.atala.prism.crypto.MerkleRoot
import io.iohk.atala.prism.crypto.Sha256
import io.iohk.atala.prism.crypto.Sha256Digest
import io.iohk.atala.prism.protos.CredentialBatchData
import pbandk.ByteArr
import pbandk.encodeToByteArray
import kotlin.js.JsExport
import kotlin.jvm.JvmStatic

/**
 * Representation of the id of a credential batch.
 * @property id in the form of a string.
 */
@JsExport
public data class CredentialBatchId private constructor(val id: String) {
    public companion object {
        private val CREDENTIAL_BATCH_ID_RE = Regex("^[0-9a-f]{64}$")

        /**
         * @return a new [CredentialBatchId] with [CredentialBatchId.id] as given @param[id].
         * @param[id] must match the following regex ^[0-9a-f]{64}$. Otherwise, @return[null].
         */
        @JvmStatic
        public fun fromString(id: String): CredentialBatchId? =
            if (id.matches(CREDENTIAL_BATCH_ID_RE))
                CredentialBatchId(id)
            else
                null

        /**
         * @return a new [CredentialBatchId] with [CredentialBatchId.id] generated by @param[digest].
         */
        @JvmStatic
        public fun fromDigest(digest: Sha256Digest): CredentialBatchId =
            CredentialBatchId(digest.hexValue)

        /**
         * @return a new [CredentialBatchId] with [CredentialBatchId.id] generated from @param[issuerDidSuffix] and @param[merkleRoot]
         */
        @JvmStatic
        public fun fromBatchData(issuerDidSuffix: String, merkleRoot: MerkleRoot): CredentialBatchId {
            val digest = Sha256
                .compute(
                    CredentialBatchData(
                        issuerDid = issuerDidSuffix,
                        merkleRoot = ByteArr(merkleRoot.hash.value)
                    ).encodeToByteArray()
                )

            return fromDigest(digest)
        }

        /**
         * @return a new [CredentialBatchId] with [CredentialBatchId.id] generated randomly.
         */
        @JvmStatic
        public fun random(): CredentialBatchId =
            CredentialBatchId(
                Sha256.compute(uuid4().toString().encodeToByteArray()).hexValue
            )
    }
}
