package io.iohk.atala.prism.api.node

import io.iohk.atala.prism.api.CredentialVerificationService
import io.iohk.atala.prism.api.VerificationResult
import io.iohk.atala.prism.api.models.*
import io.iohk.atala.prism.api.models.ProtocolVersion
import io.iohk.atala.prism.credentials.PrismCredential
import io.iohk.atala.prism.crypto.MerkleInclusionProof
import io.iohk.atala.prism.crypto.Sha256Digest
import io.iohk.atala.prism.identity.*
import io.iohk.atala.prism.protos.*
import io.iohk.atala.prism.protos.models.toModel
import pbandk.ByteArr

/**
 * @see NodePublicApi
 *
 * Implementation of [NodePublicApi] based on a remote connection.
 */
public class NodePublicApiImpl internal constructor(private val client: NodeServiceCoroutine) :
    NodePublicApi {

    /**
     * @param options gRPC options pointing to the running PRISM Node service
     */
    public constructor(options: GrpcOptions) : this(NodeServiceCoroutine.Client(GrpcClient(options)))

    private val verificationService = CredentialVerificationService(client)

    override suspend fun healthCheck() {
        client.HealthCheck(HealthCheckRequest())
    }

    override suspend fun verify(
        signedCredential: PrismCredential,
        merkleInclusionProof: MerkleInclusionProof
    ): VerificationResult =
        verificationService.verify(signedCredential, merkleInclusionProof)

    override suspend fun getDidDocument(did: PrismDid): PrismDidState {

        val (document, lastSyncedBlockTimestamp, lastUpdateOperation) =
            when (did) {
                is LongFormPrismDid -> {
                    val response = client.GetDidDocument(GetDidDocumentRequest(did = did.asCanonical().value))
                    val didData = response.document?.let {
                        DIDData(did.suffix, it.publicKeys)
                    } ?: did.toDidDataProto()
                    Triple(didData, response.lastSyncedBlockTimestamp, response.lastUpdateOperation.array)
                }
                else -> {
                    val response = client.GetDidDocument(GetDidDocumentRequest(did = did.value))
                    val didData = response.document
                    require(didData != null) {
                        "Node response does not include a valid DID document"
                    }
                    Triple(didData, response.lastSyncedBlockTimestamp, response.lastUpdateOperation.array)
                }
            }

        require(document.id == did.suffix) {
            "document.id must be a DID suffix, received: ${document.id}, actual suffix is ${did.suffix}"
        }
        val keys = document.publicKeys.map {
            val prismKeyInformation = it.toModel()
            require(prismKeyInformation != null) {
                "Node returned an invalid public key information"
            }
            prismKeyInformation
        }
        return PrismDidState(
            PrismDidDataModel(did, keys.toTypedArray()),
            if (lastUpdateOperation.isNotEmpty()) Sha256Digest.fromBytes(lastUpdateOperation) else null,
            lastSyncedBlockTimestamp
        )
    }

    override suspend fun getOperationInfo(operationId: AtalaOperationId): AtalaOperationInfo {
        val response =
            client.GetOperationInfo(GetOperationInfoRequest(ByteArr(operationId.value())))
        val transactionId: String? = if (response.transactionId.isNotEmpty()) response.transactionId else null
        return AtalaOperationInfo(
            response.operationStatus.toModel(),
            response.details,
            transactionId,
            response.lastSyncedBlockTimestamp
        )
    }

    override suspend fun getCredentialRevocationTime(
        batchId: String,
        credentialHash: Sha256Digest
    ): GetCredentialRevocationTimeResult {
        val request = GetCredentialRevocationTimeRequest(
            batchId = batchId,
            credentialHash = ByteArr(credentialHash.value)
        )
        val response = client.GetCredentialRevocationTime(request)

        val (revocationLedgerData, lastSyncedBlockTimestamp) = response

        require(lastSyncedBlockTimestamp != null) {
            "Node response does not include a valid last synchronized timestamp"
        }

        return GetCredentialRevocationTimeResult(
            revocationLedgerData?.toModel(),
            lastSyncedBlockTimestamp
        )
    }

    override suspend fun getBatchState(
        batchId: String,
    ): BatchState {
        val request = GetBatchStateRequest(batchId)
        val response = client.GetBatchState(request)

        val lastSyncedBlockTimestamp = response.lastSyncedBlockTimestamp
        require(lastSyncedBlockTimestamp != null) {
            "Node response does not include a valid last synchronized timestamp"
        }

        val publicationLedgerData = response.publicationLedgerData
        require(publicationLedgerData != null) {
            "Node response does not include a valid publication Ledger Data"
        }

        return BatchState(
            issuerDid = response.issuerDid,
            merkleRoot = Sha256Digest.fromBytes(response.merkleRoot.array),
            publicationLedgerData = publicationLedgerData.toModel(),
            revocationLedgerData = response.revocationLedgerData?.toModel(),
            issuanceHash = Sha256Digest.fromBytes(response.issuanceHash.array),
            lastSyncedBlockTimestamp = lastSyncedBlockTimestamp
        )
    }

    override suspend fun getNodeBuildInfo(): NodeBuildInfo {
        val response = client.GetNodeBuildInfo(GetNodeBuildInfoRequest())
        return NodeBuildInfo(
            response.version,
            response.scalaVersion,
            response.sbtVersion
        )
    }

    override suspend fun getNodeNetworkProtocolInfo(): NodeNetworkProtocolInfo {
        val response = client.GetNodeNetworkProtocolInfo(GetNodeNetworkProtocolInfoRequest())
        return NodeNetworkProtocolInfo(
            ProtocolVersion.fromProto(
                response.supportedNetworkProtocolVersion
                    ?: throw IllegalStateException("Node response does not include a valid supported network protocol version")
            ),
            ProtocolVersion.fromProto(
                response.currentNetworkProtocolVersion
                    ?: throw IllegalStateException("Node response does not include a valid current network protocol version")
            )
        )
    }
}
