package io.iohk.atala.prism.api.node

import io.iohk.atala.prism.api.VerificationResult
import io.iohk.atala.prism.api.models.AtalaOperationId
import io.iohk.atala.prism.api.models.AtalaOperationInfo
import io.iohk.atala.prism.api.models.ProtocolVersion
import io.iohk.atala.prism.credentials.PrismCredential
import io.iohk.atala.prism.crypto.MerkleInclusionProof
import io.iohk.atala.prism.crypto.Sha256Digest
import io.iohk.atala.prism.identity.PrismDid
import io.iohk.atala.prism.identity.PrismDidDataModel
import io.iohk.atala.prism.protos.models.LedgerData
import pbandk.wkt.Timestamp
import kotlin.js.JsExport

/**
 * @param ledgerData optional information about when the credential was revoked, if null, then the
 * credential was never revoked
 * @param lastSyncedBlockTimestamp the latest synchronized Cardano block timestamp received from
 * Node
 */
@JsExport
public data class GetCredentialRevocationTimeResult(
    val ledgerData: LedgerData?,
    val lastSyncedBlockTimestamp: Timestamp
)

/**
 * @param issuerDid issuer did of the issuer of the batch
 * @param merkleRoot merkle root of all credential issued in the batch
 * @param publicationLedgerData information about when the batch was published
 * @param revocationLedgerData optional information about when the batch was revoked, if null, then the
 * batch was never revoked
 * @param lastSyncedBlockTimestamp the latest synchronized Cardano block timestamp received from
 * Node
 */
@JsExport
public data class BatchState(
    public val issuerDid: String = "",
    public val merkleRoot: Sha256Digest,
    public val publicationLedgerData: LedgerData,
    public val revocationLedgerData: LedgerData?,
    public val issuanceHash: Sha256Digest,
    public val lastSyncedBlockTimestamp: Timestamp
)

/**
 * @param didData contains the did content
 * @param lastUpdateOperation represents the id of the last operation that modified the did
 * @param lastSyncedBlockTimestamp the latest synchronized Cardano block timestamp received from
 * Node
 */
@JsExport
public data class PrismDidState(
    val didData: PrismDidDataModel,
    val lastUpdateOperation: Sha256Digest?,
    val lastSyncedBlockTimestamp: Timestamp?
)

@JsExport
public data class NodeBuildInfo(
    val version: String,
    val scalaVersion: String,
    val sbtVersion: String
)

@JsExport
public data class NodeNetworkProtocolInfo(
    val supportedNetworkProtocolVersion: ProtocolVersion,
    val currentNetworkProtocolVersion: ProtocolVersion
)

/**
 * This class provides a way to communicate with PRISM Node via requests that do not require
 * ownership of any kind of private keys.
 */
public interface NodePublicApi {
    /**
     * Checks if the Node is healthy and, if so, finishes; throws an exception otherwise
     */
    public suspend fun healthCheck()

    /**
     * Verifies whether a given signed credential along with a corresponding Merkle proof can be
     * verified on the Node.
     *
     * @return verification result with a list of errors (empty if the verification ended
     * successfully) and the latest synchronized Cardano block timestamp if any
     */
    public suspend fun verify(
        signedCredential: PrismCredential,
        merkleInclusionProof: MerkleInclusionProof
    ): VerificationResult

    /**
     * Fetches the current version of DID document from Node.
     *
     * @param did DID of interest
     * @return the current version of DID document according to Node
     */
    public suspend fun getDidDocument(did: PrismDid): PrismDidState

    /**
     * Fetches the operation information from Node by operation identifier.
     * Information contains: status of an operation, timestamp of the last Cardano block synced with PRISM Node,
     * if status of an operation is one of AWAIT_CONFIRMATION, APPROVED_AND_APPLIED, APPROVED_AND_REJECTED
     * then operation info also contains the corresponding transaction identifier.
     *
     * @param operationId id of the operation of interest
     * @return operation information which includes status, transaction identifier, and the timestamp of the last synced block
     */
    public suspend fun getOperationInfo(operationId: AtalaOperationId): AtalaOperationInfo

    /**
     * Fetches the revocation time (if any) for a credential.
     *
     * @param batchId credential batch to which the given credential belongs to
     * @param credentialHash credential hash of the given credential
     * @return optional ledger data (null if the credential was never revoked) along with the
     * latest synchronized Cardano block timestamp
     */
    public suspend fun getCredentialRevocationTime(
        batchId: String,
        credentialHash: Sha256Digest
    ): GetCredentialRevocationTimeResult

    /**
     * Fetches the batch information of a credential batch.
     *
     * @param batchId credential batch
     * @return batch information
     */
    public suspend fun getBatchState(
        batchId: String,
    ): BatchState

    public suspend fun getNodeBuildInfo(): NodeBuildInfo

    public suspend fun getNodeNetworkProtocolInfo(): NodeNetworkProtocolInfo
}
