package io.iohk.atala.prism.api.node

import io.iohk.atala.prism.api.*
import io.iohk.atala.prism.api.models.ProtocolVersion
import io.iohk.atala.prism.credentials.CredentialBatchId
import io.iohk.atala.prism.crypto.MerkleRoot
import io.iohk.atala.prism.crypto.Sha256
import io.iohk.atala.prism.crypto.Sha256Digest
import io.iohk.atala.prism.crypto.keys.ECPrivateKey
import io.iohk.atala.prism.identity.LongFormPrismDid
import io.iohk.atala.prism.identity.PrismDid
import io.iohk.atala.prism.identity.PrismKeyInformation
import pbandk.encodeToByteArray
import kotlin.js.JsExport

/**
 * @param payload payload that can be used for making [NodeAuthApi.createDid] call
 * @param operationHash SHA256 hash of the creation operation
 */
@JsExport
public data class CreateDidInfo(
    val payload: PrismPayload,
    val operationHash: Sha256Digest
)

/**
 * @param payload payload that can be used for making [NodeAuthApi.updateDid] call
 * @param operationHash SHA256 hash of the update operation
 */
@JsExport
public data class UpdateDidInfo(
    val payload: PrismPayload,
    val operationHash: Sha256Digest
)

/**
 * @param payload payload that can be used for making [NodeAuthApi.issueCredentials] call
 * @param batchId identifier of the issued credential batch
 * @param operationHash SHA256 hash of the issuance operation
 * @param credentialsAndProofs array of signed credentials along with their Merkle proofs of
 * belonging to the given credential batch
 * @param merkleRoot Merkle root that was used for creating the issuance batch, can be combined with
 * any of the proofs from [credentialsAndProofs] for validating that the credential belongs to this
 * batch
 */
@JsExport
public data class IssueCredentialsInfo(
    val payload: PrismPayload,
    val batchId: CredentialBatchId,
    val operationHash: Sha256Digest,
    val credentialsAndProofs: Array<CredentialInfo>,
    val merkleRoot: MerkleRoot
)

/**
 * @param payload payload that can be used for making [NodeAuthApi.revokeCredentials] call
 */
@JsExport
public data class RevokeCredentialsInfo(
    val payload: PrismPayload,
    val operationHash: Sha256Digest
)

/**
 * @param payload payload that can be used for making [NodeAuthApi.protocolVersionUpdate] call
 */
@JsExport
public data class ProtocolVersionUpdateInfo(
    val payload: PrismPayload,
    val operationHash: Sha256Digest
)

/**
 * @param payload payload that can be used for making [NodeAuthApi.deactivateDID] call
 */
@JsExport
public data class DeactivateDIDInfo(
    val payload: PrismPayload,
    val operationHash: Sha256Digest
)

/**
 * This class provides a way to generate payloads on the client side that controls all private [keys]
 * related to the given [did]. None of the methods actually make the Node API calls, but rather give
 * caller a payload that can be used to invoke one of the methods from [NodeAuthApi].
 *
 * Example of usage:
 * ```
 * val masterKey = EC.generateKeyPair()
 * val did = PrismDid.buildLongFormFromMasterPublicKey(masterKey.publicKey)
 * val generator = NodePayloadGenerator(did, mapOf(PrismDid.DEFAULT_MASTER_KEY_ID, masterKey.privateKey))
 * val createDidInfo = generator.createDid()
 * ```
 *
 * @param did the DID that caller controls
 * @param keys map of key ids to the corresponding private keys
 */
@JsExport
public class NodePayloadGenerator(
    public val did: LongFormPrismDid,
    public val keys: Map<String, ECPrivateKey>
) {
    private fun findKey(keyId: String): ECPrivateKey {
        val masterKey = keys[keyId]
        require(masterKey != null) {
            "Could not find key with id '$keyId'"
        }
        return masterKey
    }

    /**
     * @param masterKeyId the id of the key to be used as the original master key for the purposes
     * of creating the DID, must be present in the [keys] field
     * @return information about DID creation that can be further used for making a Node API call
     */
    public fun createDid(masterKeyId: String = PrismDid.DEFAULT_MASTER_KEY_ID): CreateDidInfo {
        val masterKey = findKey(masterKeyId)

        val signedAtalaOperation = ProtoUtils.signedAtalaOperation(
            masterKey,
            PrismDid.DEFAULT_MASTER_KEY_ID,
            did.initialState
        )
        return CreateDidInfo(
            PrismPayload(signedAtalaOperation.encodeToByteArray()),
            did.stateHash
        )
    }

    /**
     * @param previousHash the SHA256 hash of the last operation applied to the DID (can be either
     * the original creation operation if this is the first update or the last known update
     * operation)
     * @param masterKeyId the id of the master key to be used for modifying the DID, must be present
     * in the [keys] field
     * @param keysToAdd array of keys to add to the DID, can be empty but all entries must have
     * unique ids that were not present in the DID before
     * @param keysToRevoke array of keys to revoke from the DID, can be empty but all entries must
     * represent a key that already exists in the DID. There are **no checks** that the revoked keys
     * were actually a part of the DID.
     * @return information about DID update that can be further used for making a Node API call
     */
    public fun updateDid(
        previousHash: Sha256Digest,
        masterKeyId: String,
        keysToAdd: Array<PrismKeyInformation> = emptyArray(),
        keysToRevoke: Array<String> = emptyArray()
    ): UpdateDidInfo {
        val masterKey = findKey(masterKeyId)

        val updateDidContext = NodeApiUtils.updateDidAtalaOperation(
            masterKey,
            PrismDid.DEFAULT_MASTER_KEY_ID,
            did.asCanonical(),
            previousHash,
            keysToAdd,
            keysToRevoke
        )

        return UpdateDidInfo(
            PrismPayload(updateDidContext.updateDidSignedOperation.encodeToByteArray()),
            updateDidContext.operationHash
        )
    }

    /**
     * @param issuingKeyId the id of the issuing key to be used for modifying the DID, must be
     * present in the [keys] field
     * @param credentialsClaims array of unsigned credential claims to be included in this issuance batch
     * @return information about credentials issuance that can be further used for making a Node API
     * call
     */
    public fun issueCredentials(
        issuingKeyId: String,
        credentialsClaims: Array<CredentialClaim>
    ): IssueCredentialsInfo {
        val issuingKey = findKey(issuingKeyId)

        val issueBatchContext = CredentialBatches.createBatchAtalaOperation(
            did.asCanonical(),
            issuingKeyId,
            issuingKey,
            credentialsClaims.toList()
        )

        return IssueCredentialsInfo(
            PrismPayload(issueBatchContext.signedAtalaOperation.encodeToByteArray()),
            issueBatchContext.batchId,
            issueBatchContext.issuanceOperationHash,
            issueBatchContext.credentialsAndProofs.toTypedArray(),
            issueBatchContext.merkleRoot
        )
    }

    /**
     * @param revocationKeyId the id of the revocation key to be used for modifying the DID, must be
     * present in the [keys] field
     * @param previousOperationHash the SHA256 hash of the previous known operation applied to this
     * credential batch (either the original issuance operation returned by
     * [issueCredentials] or previous applied revocation operation)
     * @param batchId the id of the credential batch
     * @param credentialsToRevoke array of credential hashes to be revoked, if empty, then all
     * credentials from the credential batch will be revoked. If non-empty, there are **no
     * checks** that the given credential hashes actually were issued in the credential batch.
     * @return information about credentials revocation that can be further used for making a Node
     * API call
     */
    public fun revokeCredentials(
        revocationKeyId: String,
        previousOperationHash: Sha256Digest,
        batchId: String,
        credentialsToRevoke: Array<Sha256Digest> = emptyArray()
    ): RevokeCredentialsInfo {
        val revocationKey = findKey(revocationKeyId)

        val revokeCredentialsSignedOperation = ProtoUtils.revokeCredentialsSignedOperation(
            revocationKeyId,
            revocationKey,
            previousOperationHash,
            batchId,
            credentialsToRevoke.toList()
        )

        return RevokeCredentialsInfo(
            PrismPayload(revokeCredentialsSignedOperation.encodeToByteArray()),
            operationHash = Sha256.compute(revokeCredentialsSignedOperation.operation!!.encodeToByteArray())
        )
    }

    /**
     * @param masterKeyId the id of the master key to be used for issuing a protocol version update
     * @param versionName an optional name of the version
     * @param protocolVersion pair representing major and minor versions
     * @param effectiveSinceBlockIndex a Cardano block index since the update gets effective
     * @return information about protocol update operation that can be further used for making a Node
     * API call
     */
    public fun protocolVersionUpdate(
        masterKeyId: String,
        versionName: String?,
        protocolVersion: ProtocolVersion,
        effectiveSinceBlockIndex: Int
    ): ProtocolVersionUpdateInfo {
        val masterKey = findKey(masterKeyId)

        val signedProtocolUpdateOp = ProtoUtils.protocolUpdateSignedOperation(
            masterKeyId,
            masterKey,
            did.asCanonical(),
            versionName,
            protocolVersion,
            effectiveSinceBlockIndex
        )

        return ProtocolVersionUpdateInfo(
            PrismPayload(signedProtocolUpdateOp.encodeToByteArray()),
            operationHash = Sha256.compute(signedProtocolUpdateOp.operation!!.encodeToByteArray())
        )
    }

    public fun deactivateDid(
        previousOperationDigest: Sha256Digest,
        masterKeyId: String
    ): DeactivateDIDInfo {
        val masterKey = findKey(masterKeyId)
        val signedOp = ProtoUtils.deactivateDIDOperation(
            masterKeyId,
            masterKey,
            did.asCanonical(),
            previousOperationDigest
        )

        return DeactivateDIDInfo(
            PrismPayload(signedOp.encodeToByteArray()),
            operationHash = Sha256.compute(signedOp.operation!!.encodeToByteArray())
        )
    }
}
