package io.iohk.atala.prism.api.node

import com.ionspin.kotlin.bignum.integer.BigInteger
import com.ionspin.kotlin.bignum.integer.Sign
import io.iohk.atala.prism.api.common.AuthDetails
import io.iohk.atala.prism.api.common.authenticatedApiCall
import io.iohk.atala.prism.protos.*
import io.ktor.utils.io.core.*

public class NodeExplorerAuthApiImpl internal constructor(
    private val client: NodeExplorerServiceCoroutine
) :
    NodeExplorerApi,
    Closeable {

    public constructor(
        options: GrpcOptions
    ) : this(NodeExplorerServiceCoroutine.Client(GrpcClient(options)))

    override suspend fun getScheduledOperations(
        operationsKind: AtalaOperationType?,
        authDetails: AuthDetails
    ): List<SignedAtalaOperation> {
        val opKind = operationsKind?.toProto() ?: GetScheduledOperationsRequest.OperationType.ANY_OPERATION_TYPE
        return authenticatedApiCall(
            GetScheduledOperationsRequest(opKind),
            authDetails,
            isCanonical = false,
        ) { request, metadata ->
            client.GetScheduledOperationsAuth(
                request,
                metadata
            )
        }.scheduledOperations
    }

    override suspend fun getWalletTransactions(
        transactionStatus: WalletTransactionStatus?,
        lastSeenTransactionId: String?,
        limit: Int,
        authDetails: AuthDetails
    ): List<TransactionState> {
        if (limit > 50)
            throw IllegalArgumentException("limit value should be up to 50")

        suspend fun reqWithStatus(
            status: WalletTransactionStatus,
            lastSeenTxId: String? = lastSeenTransactionId,
            lim: Int = limit
        ): List<TransactionState> {
            return authenticatedApiCall(
                GetWalletTransactionsRequest(
                    status.toProto(),
                    lastSeenTxId ?: "",
                    lim
                ),
                authDetails,
                isCanonical = false,
            ) { request, metadata ->
                client.GetWalletTransactionsPaginatedAuth(
                    request,
                    metadata
                )
            }.transactions.map { TransactionState.fromProto(it, status) }
        }

        return transactionStatus?.let { reqWithStatus(transactionStatus) } ?: run {
            val ongoing = reqWithStatus(WalletTransactionStatus.Companion.OngoingTransaction)

            val confirmed =
                if (ongoing.size < limit) {
                    val (restLimit, lastSeenTxId) =
                        if (ongoing.isEmpty()) Pair(limit, lastSeenTransactionId)
                        else Pair(limit - ongoing.size, null)

                    reqWithStatus(
                        WalletTransactionStatus.Companion.ConfirmedTransaction,
                        lastSeenTxId,
                        restLimit
                    )
                } else emptyList()

            // Filtering out trick here is only needed to avoid a rare situation when
            // between two Node requests above an ongoing transaction gets confirmed
            val ongoingSet = ongoing.map { it.transactionId }.toSet()
            return ongoing + confirmed.filter { !ongoingSet.contains(it.transactionId) }
        }
    }

    override suspend fun getWalletBalance(authDetails: AuthDetails): BigInteger {
        val balanceAsByteArray = authenticatedApiCall(
            GetWalletBalanceRequest(),
            authDetails,
            isCanonical = false
        ) { request, metadata ->
            client.GetWalletBalanceAuth(request, metadata)
        }.balance.array

        require(balanceAsByteArray.isNotEmpty()) {
            "Expected Balance Byte Array not to be empty"
        }

        return BigInteger.fromByteArray(balanceAsByteArray, sign = Sign.POSITIVE)
    }

    override fun close() {
        client.close()
    }
}
