package io.iohk.atala.prism.api.node

import com.ionspin.kotlin.bignum.integer.BigInteger
import io.iohk.atala.prism.api.common.AuthDetails
import io.iohk.atala.prism.protos.GetScheduledOperationsRequest
import io.iohk.atala.prism.protos.GetWalletTransactionsRequest
import io.iohk.atala.prism.protos.SignedAtalaOperation
import io.iohk.atala.prism.protos.TransactionInfo
import io.iohk.atala.prism.protos.models.LedgerEnum
import io.iohk.atala.prism.protos.models.toModel
import kotlinx.datetime.Instant
import kotlin.js.JsExport

/**
 * Operation type identifying requested operations
 */
@JsExport
public sealed class AtalaOperationType {
    public abstract fun toProto(): GetScheduledOperationsRequest.OperationType

    public object CreateDidType : AtalaOperationType() {
        override fun toProto(): GetScheduledOperationsRequest.OperationType =
            GetScheduledOperationsRequest.OperationType.CREATE_DID_OPERATION_OPERATION_TYPE
    }

    public object UpdateDidType : AtalaOperationType() {
        override fun toProto(): GetScheduledOperationsRequest.OperationType =
            GetScheduledOperationsRequest.OperationType.UPDATE_DID_OPERATION_OPERATION_TYPE
    }

    public object IssueCredentialBatchType : AtalaOperationType() {
        override fun toProto(): GetScheduledOperationsRequest.OperationType =
            GetScheduledOperationsRequest.OperationType.ISSUE_CREDENTIAL_BATCH_OPERATION_TYPE
    }

    public object RevokeCredentialsType : AtalaOperationType() {
        override fun toProto(): GetScheduledOperationsRequest.OperationType =
            GetScheduledOperationsRequest.OperationType.REVOKE_CREDENTIALS_OPERATION_TYPE
    }

    public object ProtocolVersionUpdateType : AtalaOperationType() {
        override fun toProto(): GetScheduledOperationsRequest.OperationType =
            GetScheduledOperationsRequest.OperationType.PROTOCOL_VERSION_UPDATE_OPERATION_TYPE
    }

    public companion object {
        public fun fromString(s: String): AtalaOperationType {
            return when (s) {
                "create-did" -> CreateDidType
                "update-did" -> UpdateDidType
                "issue-credential-batch" -> IssueCredentialBatchType
                "revoke-credentials" -> RevokeCredentialsType
                "protocol-update" -> ProtocolVersionUpdateType
                else -> throw IllegalArgumentException("Invalid operation type: $s")
            }
        }
    }
}

/**
 * Blockchain transaction status.
 */

@JsExport
public sealed class WalletTransactionStatus {
    public abstract fun toProto(): GetWalletTransactionsRequest.TransactionState

    public companion object {
        public object OngoingTransaction : WalletTransactionStatus() {
            override fun toProto(): GetWalletTransactionsRequest.TransactionState =
                GetWalletTransactionsRequest.TransactionState.ONGOING
        }

        public object ConfirmedTransaction : WalletTransactionStatus() {
            override fun toProto(): GetWalletTransactionsRequest.TransactionState =
                GetWalletTransactionsRequest.TransactionState.CONFIRMED
        }

        public fun fromString(s: String): WalletTransactionStatus {
            return when (s) {
                "ongoing" -> OngoingTransaction
                "confirmed" -> ConfirmedTransaction
                else -> throw IllegalArgumentException("Invalid wallet transaction status: $s")
            }
        }
    }
}

/**
 * @param blockLevel level of the block in the blockchain
 * @param transactionIndex index of the transaction within the block
 * @param blockTimestamp timestamp of the block
 */
@JsExport
public data class BlockInfo(
    val blockLevel: Int,
    val transactionIndex: Int,
    val blockTimestamp: Long
)

/**
 * @param transactionId transaction id in the blockchain
 * @param ledger represents the underlying ledger of the transaction
 * @param status indicates transaction status
 */
@JsExport
public data class TransactionState(
    val transactionId: String,
    val ledger: LedgerEnum,
    val blockInfo: BlockInfo?,
    val status: WalletTransactionStatus
) {
    public companion object {
        public fun fromProto(info: TransactionInfo, status: WalletTransactionStatus): TransactionState {
            return TransactionState(
                transactionId = info.transactionId,
                ledger = info.ledger.toModel(),
                blockInfo = info.block?.let { b ->
                    val instant = Instant.fromEpochSeconds(b.timestamp?.seconds!!, b.timestamp?.nanos!!)
                    BlockInfo(b.number, b.index, instant.toEpochMilliseconds())
                },
                status = status
            )
        }
    }
}

/**
 * This class provides a way for Node explorer operator
 * to get access to the Node internal data like metrics, statistics, queues and statuses.
 */
public interface NodeExplorerApi {

    /**
     * Fetches operations which have been scheduled but not confirmed yet.
     * The method returns a list of SignedAtalaOperation which might be accessed
     * only with @PrismSdkInternal annotation as this API is experimental and
     * not for public clients' usage.
     *
     * @param operationsKind optional type of operations which should be returned,
     * if not provided all the operations will be returned
     * @param authDetails Node explorer credentials in order to authenticate a request with a signature
     * @return scheduled operations known to a node
     */
    public suspend fun getScheduledOperations(
        operationsKind: AtalaOperationType? = null,
        authDetails: AuthDetails
    ): List<SignedAtalaOperation>

    /**
     * Fetches Cardano transactions which contain batches of Atala operations.
     *
     * @param transactionStatus transaction status, either ongoing (not confirmed yet) or confirmed
     * @param lastSeenTransactionId optional transaction id which has been received last
     * @param limit amount of operations to return if exist
     * @param authDetails Node explorer credentials in order to authenticate a request with a signature
     * @return transactions of the given status
     */
    public suspend fun getWalletTransactions(
        transactionStatus: WalletTransactionStatus? = null,
        lastSeenTransactionId: String? = null,
        limit: Int = 50,
        authDetails: AuthDetails
    ): List<TransactionState>

    /**
     * Returns balance of the wallet connected to the node
     *
     * @param authDetails Node explorer credentials in order to authenticate a request with a signature
     * @return wallet balance denominated in lovelaces
     */
    public suspend fun getWalletBalance(authDetails: AuthDetails): BigInteger
}
