package io.iohk.atala.prism.api.node

import io.iohk.atala.prism.api.PrismPayload
import io.iohk.atala.prism.api.common.DidValidation
import io.iohk.atala.prism.api.models.AtalaOperationId
import io.iohk.atala.prism.api.models.ProtocolVersion
import io.iohk.atala.prism.crypto.MerkleRoot
import io.iohk.atala.prism.crypto.Sha256Digest
import io.iohk.atala.prism.crypto.util.BytesOps
import io.iohk.atala.prism.identity.CanonicalPrismDid
import io.iohk.atala.prism.identity.LongFormPrismDid
import io.iohk.atala.prism.identity.PrismKeyInformation
import io.iohk.atala.prism.identity.toProto
import io.iohk.atala.prism.protos.*
import io.ktor.utils.io.core.*
import pbandk.ByteArr
import pbandk.decodeFromByteArray

/**
 * @see NodeAuthApi
 *
 * Implementation of [NodeAuthApi] based on a remote connection.
 */
public class NodeAuthApiImpl internal constructor(
    private val client: NodeServiceCoroutine,
) : NodeAuthApi, NodePublicApi by NodePublicApiImpl(client), Closeable {

    /**
     * @param options gRPC options pointing to the running PRISM Node service
     */
    public constructor(
        options: GrpcOptions
    ) : this(NodeServiceCoroutine.Client(GrpcClient(options)))

    private fun toAtalaOperationId(byteArr: ByteArr): AtalaOperationId =
        AtalaOperationId.fromBytes(byteArr.array)

    private fun extractSingleOperationId(response: ScheduleOperationsResponse): AtalaOperationId {
        require(response.outputs.size == 1) {
            "Invalid Node response, expecting to get 1 operation id but instead got: {${response.outputs.joinToString(", ")}}"
        }
        return toAtalaOperationId(response.outputs[0].operationId!!)
    }

    private inline fun <reified K> extractOperation(
        payload: PrismPayload
    ): Pair<SignedAtalaOperation, K> {
        val signedAtalaOperation = SignedAtalaOperation.decodeFromByteArray(payload.value)
        when (val operation = signedAtalaOperation.operation?.operation?.value) {
            is K -> return Pair(signedAtalaOperation, operation)
            else -> throw IllegalArgumentException("Payload does not contain Atala operation of the expected type")
        }
    }

    override suspend fun createDid(
        payload: PrismPayload,
        did: LongFormPrismDid?,
        masterKeyId: String?
    ): AtalaOperationId {
        val signedAtalaOperation = DidValidation.verifyPayload(did, masterKeyId, payload)

        val request = ScheduleOperationsRequest(listOf(signedAtalaOperation))
        val response = client.ScheduleOperations(request)

        return extractSingleOperationId(response)
    }

    override suspend fun updateDid(
        payload: PrismPayload,
        did: CanonicalPrismDid?,
        masterKeyId: String?,
        previousOperationHash: Sha256Digest?,
        keysToAdd: Array<PrismKeyInformation>?,
        keysToRevoke: Array<String>?
    ): AtalaOperationId {
        val (signedAtalaOperation, updateDidOperation) =
            extractOperation<UpdateDIDOperation>(payload)

        if (did != null) {
            require(updateDidOperation.id == did.suffix) {
                "Expected DID '${did.suffix}', but got '${updateDidOperation.id}' instead"
            }
        }

        if (masterKeyId != null) {
            require(signedAtalaOperation.signedWith == masterKeyId) {
                "The payload was signed with key '${signedAtalaOperation.signedWith}', expected '$masterKeyId'"
            }
        }

        if (previousOperationHash != null) {
            val calculatedPreviousOperationHash =
                Sha256Digest.fromBytes(updateDidOperation.previousOperationHash.array)
            require(calculatedPreviousOperationHash == previousOperationHash) {
                "Calculated previous operation hash (${calculatedPreviousOperationHash.hexValue}) does not match to the passed one (${previousOperationHash.hexValue})"
            }
        }
        keysToAdd?.forEach { keyInfo ->
            require(
                updateDidOperation.actions.find { it.addKey?.key == keyInfo.toProto() } != null
            ) {
                "Expected payload to contain an action to add key '${keyInfo.didPublicKey.id}'"
            }
        }
        keysToRevoke?.forEach { keyId ->
            require(
                updateDidOperation.actions.find { it.removeKey?.keyId == keyId } != null
            ) {
                "Expected payload to contain an action to revoke key '$keyId'"
            }
        }

        val request = ScheduleOperationsRequest(listOf(signedAtalaOperation))
        val response = client.ScheduleOperations(request)

        return extractSingleOperationId(response)
    }

    override suspend fun issueCredentials(
        payload: PrismPayload,
        did: CanonicalPrismDid?,
        issuingKeyId: String?,
        merkleRoot: MerkleRoot?
    ): AtalaOperationId {
        val (signedAtalaOperation, issueCredentialBatchOperation) =
            extractOperation<IssueCredentialBatchOperation>(payload)

        if (issuingKeyId != null) {
            require(issuingKeyId == signedAtalaOperation.signedWith) {
                "The payload was signed with key '${signedAtalaOperation.signedWith}', expected '$issuingKeyId'"
            }
        }
        if (did != null) {
            val calculatedIssuerDid = issueCredentialBatchOperation.credentialBatchData?.issuerDid
            require(calculatedIssuerDid == did.asCanonical().suffix) {
                "Expected credentials to be issued by DID '${did.asCanonical().suffix}', but got '$calculatedIssuerDid' instead"
            }
        }
        if (merkleRoot != null) {
            val calculatedMerkleRoot =
                issueCredentialBatchOperation.credentialBatchData?.merkleRoot?.array
                    ?: byteArrayOf()
            require(calculatedMerkleRoot.contentEquals(merkleRoot.hash.value)) {
                "Expected to find Merkle root '${merkleRoot.hash.hexValue}', but got" +
                    "'${BytesOps.bytesToHex(calculatedMerkleRoot)}' instead"
            }
        }

        val request = ScheduleOperationsRequest(listOf(signedAtalaOperation))
        val response = client.ScheduleOperations(request)

        return extractSingleOperationId(response)
    }

    override suspend fun revokeCredentials(
        payload: PrismPayload,
        did: CanonicalPrismDid?,
        revocationKeyId: String?,
        previousOperationHash: Sha256Digest?,
        batchId: String?,
        credentialsToRevoke: Array<Sha256Digest>?
    ): AtalaOperationId {
        val (signedAtalaOperation, revokeCredentialsOperation) =
            extractOperation<RevokeCredentialsOperation>(payload)

        if (revocationKeyId != null) {
            require(revocationKeyId == signedAtalaOperation.signedWith) {
                "The payload was signed with key '${signedAtalaOperation.signedWith}', expected '$revocationKeyId'"
            }
        }
        if (previousOperationHash != null) {
            val calculatedPreviousOperationHash =
                Sha256Digest.fromBytes(revokeCredentialsOperation.previousOperationHash.array)
            require(calculatedPreviousOperationHash == previousOperationHash) {
                "Calculated previous operation hash (${calculatedPreviousOperationHash.hexValue}) does not match to the passed one (${previousOperationHash.hexValue})"
            }
        }
        if (batchId != null) {
            require(revokeCredentialsOperation.credentialBatchId == batchId) {
                "Expected to find credential batch ID '$batchId', but found '${revokeCredentialsOperation.credentialBatchId}' instead"
            }
        }
        if (credentialsToRevoke != null) {
            val calculatedCredentialsToRevoke =
                revokeCredentialsOperation.credentialsToRevoke.map { Sha256Digest.fromBytes(it.array) }
            require(calculatedCredentialsToRevoke == credentialsToRevoke.toList()) {
                "Calculated revoked credentials do not match the passed ones"
            }
        }

        val request = ScheduleOperationsRequest(listOf(signedAtalaOperation))
        val response = client.ScheduleOperations(request)

        return extractSingleOperationId(response)
    }

    override suspend fun protocolVersionUpdate(
        payload: PrismPayload,
        did: CanonicalPrismDid?,
        masterKeyId: String?,
        versionName: String?,
        protocolVersion: ProtocolVersion?,
        effectiveSinceBlockIndex: Int?
    ): AtalaOperationId {
        val (signedAtalaOperation, protocolUpdateOperation) =
            extractOperation<ProtocolVersionUpdateOperation>(payload)

        if (masterKeyId != null) {
            require(masterKeyId == signedAtalaOperation.signedWith) {
                "The payload was signed with key '${signedAtalaOperation.signedWith}', expected '$masterKeyId'"
            }
        }

        if (versionName != null) {
            require(protocolUpdateOperation.version?.versionName == versionName) {
                "Expected to find version name: '$versionName', but found '${protocolUpdateOperation.version?.versionName}' instead"
            }
        }

        if (protocolVersion != null) {
            require(
                protocolUpdateOperation.version?.protocolVersion?.majorVersion == protocolVersion.majorVersion &&
                    protocolUpdateOperation.version?.protocolVersion?.minorVersion == protocolVersion.minorVersion
            ) {
                "Expected to find version: '$protocolVersion', but found '${protocolUpdateOperation.version?.protocolVersion?.majorVersion}.${protocolUpdateOperation.version?.protocolVersion?.minorVersion}' instead"
            }
        }

        if (effectiveSinceBlockIndex != null) {
            require(protocolUpdateOperation.version?.effectiveSince == effectiveSinceBlockIndex) {
                "Expected to find effectiveSince: '$effectiveSinceBlockIndex', but found '${protocolUpdateOperation.version?.effectiveSince}' instead"
            }
        }

        val request = ScheduleOperationsRequest(listOf(signedAtalaOperation))
        val response = client.ScheduleOperations(request)

        return extractSingleOperationId(response)
    }

    override suspend fun deactivateDID(
        payload: PrismPayload,
        did: CanonicalPrismDid?,
        masterKeyId: String?,
        previousOperationHash: Sha256Digest?
    ): AtalaOperationId {
        val (signedAtalaOperation, deactivateDIDOperation) =
            extractOperation<DeactivateDIDOperation>(payload)

        if (did != null) {
            require(deactivateDIDOperation.id == did.suffix) {
                "Expected DID '${did.suffix}', but got '${deactivateDIDOperation.id}' instead"
            }
        }

        if (masterKeyId != null) {
            require(signedAtalaOperation.signedWith == masterKeyId) {
                "The payload was signed with key '${signedAtalaOperation.signedWith}', expected '$masterKeyId'"
            }
        }

        if (previousOperationHash != null) {
            val calculatedPreviousOperationHash =
                Sha256Digest.fromBytes(deactivateDIDOperation.previousOperationHash.array)
            require(calculatedPreviousOperationHash == previousOperationHash) {
                "Calculated previous operation hash (${calculatedPreviousOperationHash.hexValue}) does not match to the passed one (${previousOperationHash.hexValue})"
            }
        }

        val request = ScheduleOperationsRequest(listOf(signedAtalaOperation))
        val response = client.ScheduleOperations(request)

        return extractSingleOperationId(response)
    }

    public override fun close() {
        client.close()
    }
}
