package io.iohk.atala.prism.api.node

import io.iohk.atala.prism.api.PrismPayload
import io.iohk.atala.prism.api.models.AtalaOperationId
import io.iohk.atala.prism.api.models.ProtocolVersion
import io.iohk.atala.prism.crypto.MerkleRoot
import io.iohk.atala.prism.crypto.Sha256Digest
import io.iohk.atala.prism.identity.CanonicalPrismDid
import io.iohk.atala.prism.identity.LongFormPrismDid
import io.iohk.atala.prism.identity.PrismKeyInformation

/**
 * This class provides a way to communicate with PRISM Node via requests that require ownership
 * of private keys belonging to a DID.
 *
 * In order to use all methods of this class, caller must have an instance of [PrismPayload] that
 * was produced by a corresponding call to one of [NodePayloadGenerator] methods.
 */
public interface NodeAuthApi : NodePublicApi {
    /**
     * Publishes the given DID on chain.
     *
     * @param payload PRISM payload produced by [NodePayloadGenerator.createDid]
     * @param did DID to be created, if not null, the payload will be checked against it and an
     * exception will be thrown in case it does not match to what the payload is trying to create
     * @param masterKeyId key id of the original DID master key, if not null, the payload will be
     * checked against it and an exception will be thrown in case it does not match to which key
     * the payload is trying to use
     */
    public suspend fun createDid(
        payload: PrismPayload,
        did: LongFormPrismDid? = null,
        masterKeyId: String? = null
    ): AtalaOperationId

    /**
     * Updates the DID document that was previously published on chain via [createDid].
     *
     * @param payload PRISM payload produced by [NodePayloadGenerator.updateDid]
     * @param did DID to be updated, if not null, the payload will be checked against it and an
     * exception will be thrown in case it does not match to what the payload is trying to update
     * @param masterKeyId key id of the master key to be used for updating the DID, if not null, the
     * payload will be checked against it and an exception will be thrown in case it does not match
     * to which key the payload is trying to use
     * @param previousOperationHash the SHA256 hash of the last operation applied to the DID (can
     * be either the original creation operation if this is the first update or the last known
     * update operation), if not null, the payload will be checked against it
     * @param keysToAdd array of keys to add to the DID, if not null, the payload will be checked
     * against it
     * @param keysToRevoke array of keys to revoke from the DID, if not null, the payload will be
     * checked against it
     */
    public suspend fun updateDid(
        payload: PrismPayload,
        did: CanonicalPrismDid? = null,
        masterKeyId: String? = null,
        previousOperationHash: Sha256Digest? = null,
        keysToAdd: Array<PrismKeyInformation>? = null,
        keysToRevoke: Array<String>? = null
    ): AtalaOperationId

    /**
     * Issues a batch of credentials and publishes it on chain.
     *
     * @param payload PRISM payload produced by [NodePayloadGenerator.issueCredentials]
     * @param did DID to be used for issuing credentials, if not null, the payload will be checked
     * against it and an exception will be thrown in case it does not match to what the payload is
     * trying to use
     * @param issuingKeyId key id of the issuing key to be used for issuing the credential batch,
     * if not null, the payload will be checked against it and an exception will be thrown in case
     * it does not match to which key the payload is trying to use
     * @param merkleRoot credential batch Merkle root produced by
     * [NodePayloadGenerator.issueCredentials], if not null, the payload will be checked against it
     */
    public suspend fun issueCredentials(
        payload: PrismPayload,
        did: CanonicalPrismDid? = null,
        issuingKeyId: String? = null,
        merkleRoot: MerkleRoot? = null
    ): AtalaOperationId

    /**
     * Revoke (some or all) credentials belonging to a batch that has been published on chain via
     * [issueCredentials].
     *
     * @param payload PRISM payload produced by [NodePayloadGenerator.revokeCredentials]
     * @param did DID to be used for revoking credentials, if not null, the payload will be checked
     * against it and an exception will be thrown in case it does not match to what the payload is
     * trying to use
     * @param revocationKeyId key id of the revocation key to be used for issuing the credential
     * batch, if not null, the payload will be checked against it and an exception will be thrown in
     * case it does not match to which key the payload is trying to use
     * @param previousOperationHash the SHA256 hash of the previous known operation applied to this
     * credential batch (either the original issuance operation returned by
     * [NodePayloadGenerator.issueCredentials] or previous applied revocation operation)
     * @param batchId the id of the credential batch, if not null, the payload will be checked
     * against it
     * @param credentialsToRevoke array of credential hashes to be revoked, if not null, the payload
     * will be checked against it. In case if some credentials have been revoked before, the operation
     * will be still considered valid.
     */
    public suspend fun revokeCredentials(
        payload: PrismPayload,
        did: CanonicalPrismDid? = null,
        revocationKeyId: String? = null,
        previousOperationHash: Sha256Digest? = null,
        batchId: String? = null,
        credentialsToRevoke: Array<Sha256Digest>? = null
    ): AtalaOperationId

    /**
     * Propose new version of the Prism Protocol.
     *
     * @param payload PRISM payload produced by [NodePayloadGenerator.protocolVersionUpdate]
     */
    public suspend fun protocolVersionUpdate(
        payload: PrismPayload,
        did: CanonicalPrismDid? = null,
        masterKeyId: String? = null,
        versionName: String? = null,
        protocolVersion: ProtocolVersion? = null,
        effectiveSinceBlockIndex: Int? = null
    ): AtalaOperationId

    /**
     * Deactivate the DID, what causes revocation of all its existing keys.
     * After deactivation, it's impossible to interact with DID document in any way.
     *
     * @param payload PRISM payload produced by [NodePayloadGenerator.deactivateDid]
     * @param did DID to be deactivated, if not null, the payload will be checked against it and an
     * exception will be thrown in case it does not match to what the payload is trying to deactivate
     * @param masterKeyId key id of the master key to be used for deactivating the DID, if not null, the
     * payload will be checked against it and an exception will be thrown in case it does not match
     * to which key the payload is trying to use
     * @param previousOperationHash the SHA256 hash of the last operation applied to the DID (can
     * be either the original creation operation if this is the first update or the last known
     * update operation), if not null, the payload will be checked against it
     */
    public suspend fun deactivateDID(
        payload: PrismPayload,
        did: CanonicalPrismDid? = null,
        masterKeyId: String? = null,
        previousOperationHash: Sha256Digest? = null
    ): AtalaOperationId
}
