package io.iohk.atala.prism.api.node

import io.iohk.atala.prism.api.KeyGenerator
import io.iohk.atala.prism.api.ProtoUtils
import io.iohk.atala.prism.crypto.Sha256
import io.iohk.atala.prism.crypto.Sha256Digest
import io.iohk.atala.prism.crypto.derivation.KeyDerivation
import io.iohk.atala.prism.crypto.derivation.MnemonicCode
import io.iohk.atala.prism.crypto.keys.ECKeyPair
import io.iohk.atala.prism.crypto.keys.ECPrivateKey
import io.iohk.atala.prism.crypto.keys.ECPublicKey
import io.iohk.atala.prism.crypto.keys.toProto
import io.iohk.atala.prism.identity.*
import io.iohk.atala.prism.protos.*
import pbandk.ByteArr
import pbandk.encodeToByteArray

internal data class CreateDidContext(
    val unpublishedDid: LongFormPrismDid,
    val createDidSignedOperation: SignedAtalaOperation,
    val operationHash: Sha256Digest,
    val masterKeyPair: ECKeyPair
)

internal data class UpdateDidContext(
    val updateDidSignedOperation: SignedAtalaOperation,
    val operationHash: Sha256Digest
)

internal object NodeApiUtils {
    internal data class AtalaOperationWithHash(
        val atalaOperation: AtalaOperation,
        val atalaOperationHash: Sha256Digest
    )

    /**
     * Creating a DID
     *
     * @param mnemonic list of 24 words that will used in generating the binary seed from [io.iohk.atala.prism.crypto.derivation.KeyDerivation.binarySeed]
     * @param didIndex index of this particular DID among all DIDs used by the user (starts from 0 and counts up every time user needs a new DID)
     * @param passphrase a password that is required for generating the binary seed from [io.iohk.atala.prism.crypto.derivation.KeyDerivation.binarySeed]
     * @return DID context
     */
    fun createDidFromMnemonic(
        mnemonic: MnemonicCode,
        didIndex: Int,
        passphrase: String = ""
    ): CreateDidContext {
        val seed = KeyDerivation.binarySeed(mnemonic, passphrase)
        val masterECKeyPair = KeyGenerator.deriveKeyFromFullPath(seed, didIndex, MasterKeyUsage, 0)
        val did = PrismDid.buildLongFormFromMasterPublicKey(masterECKeyPair.publicKey)
        val (atalaOp, operationHash) = createDidAtalaOperation(masterECKeyPair.publicKey)
        return CreateDidContext(
            unpublishedDid = did,
            createDidSignedOperation = ProtoUtils.signedAtalaOperation(
                masterECKeyPair.privateKey,
                PrismDid.DEFAULT_MASTER_KEY_ID,
                atalaOp
            ),
            operationHash = operationHash,
            masterKeyPair = masterECKeyPair
        )
    }

    /**
     * DISCLAIMER: This is an experimental function which could change or be removed without notice
     *
     * This is the same as [createDidFromMnemonic] but it includes the issuance and revocation keys
     * in the initial DID.
     *
     * The intended usage is for the browser wallet which requires to create a DID with these keys.
     *
     * @param mnemonic list of 24 words that will used in generating the binary seed from [io.iohk.atala.prism.crypto.derivation.KeyDerivation.binarySeed]
     * @param didIndex index of this particular DID among all DIDs used by the user (starts from 0 and counts up every time user needs a new DID)
     * @param passphrase a password that is required for generating the binary seed from [io.iohk.atala.prism.crypto.derivation.KeyDerivation.binarySeed]
     * @return DID context
     */
    fun createExperimentalDidFromMnemonic(
        mnemonic: MnemonicCode,
        didIndex: Int,
        passphrase: String = ""
    ): CreateDidContext {
        val seed = KeyDerivation.binarySeed(mnemonic, passphrase)
        val masterECKeyPair = KeyGenerator.deriveKeyFromFullPath(seed, didIndex, MasterKeyUsage, 0)
        val issuingECKeyPair = KeyGenerator.deriveKeyFromFullPath(seed, didIndex, IssuingKeyUsage, 0)
        val revocationECKeyPair = KeyGenerator.deriveKeyFromFullPath(seed, didIndex, RevocationKeyUsage, 0)
        val did = PrismDid.buildExperimentalLongFormFromKeys(
            masterKey = masterECKeyPair.publicKey,
            issuingKey = issuingECKeyPair.publicKey,
            revocationKey = revocationECKeyPair.publicKey
        )
        val (atalaOp, operationHash) = createDidAtalaOperation(
            masterKey = masterECKeyPair.publicKey,
            issuingKey = issuingECKeyPair.publicKey,
            revocationKey = revocationECKeyPair.publicKey
        )
        return CreateDidContext(
            unpublishedDid = did,
            createDidSignedOperation = ProtoUtils.signedAtalaOperation(
                masterECKeyPair.privateKey,
                PrismDid.DEFAULT_MASTER_KEY_ID,
                atalaOp
            ),
            operationHash = operationHash,
            masterKeyPair = masterECKeyPair
        )
    }

    fun createDidAtalaOperation(
        masterKey: ECPublicKey,
        issuingKey: ECPublicKey? = null,
        revocationKey: ECPublicKey? = null
    ): AtalaOperationWithHash {
        val masterKeyPublicKey =
            listOf(
                PublicKey(
                    id = PrismDid.DEFAULT_MASTER_KEY_ID,
                    usage = KeyUsage.MASTER_KEY,
                    keyData = PublicKey.KeyData.CompressedEcKeyData(masterKey.toProto())
                )
            )
        val issuingKeyPublicKey =
            issuingKey?.let {
                listOf(
                    PublicKey(
                        id = PrismDid.DEFAULT_ISSUING_KEY_ID,
                        usage = KeyUsage.ISSUING_KEY,
                        keyData = PublicKey.KeyData.CompressedEcKeyData(it.toProto())
                    )
                )
            }.orEmpty()

        val revocationKeyKeyPublicKey =
            revocationKey?.let {
                listOf(
                    PublicKey(
                        id = PrismDid.DEFAULT_REVOCATION_KEY_ID,
                        usage = KeyUsage.REVOCATION_KEY,
                        keyData = PublicKey.KeyData.CompressedEcKeyData(it.toProto())
                    )
                )
            }.orEmpty()

        val createDidOp = CreateDIDOperation(
            didData = CreateDIDOperation.DIDCreationData(
                publicKeys = masterKeyPublicKey + issuingKeyPublicKey + revocationKeyKeyPublicKey
            )
        )
        val atalaOp = AtalaOperation(operation = AtalaOperation.Operation.CreateDid(createDidOp))
        val operationBytes = atalaOp.encodeToByteArray()
        val operationHash = Sha256.compute(operationBytes)
        return AtalaOperationWithHash(atalaOp, operationHash)
    }

    fun updateDidAtalaOperation(
        signingPrivateKey: ECPrivateKey,
        signingKeyId: String,
        did: PrismDid,
        previousHash: Sha256Digest,
        keysToAdd: Array<PrismKeyInformation> = emptyArray(),
        keysToRevoke: Array<String> = emptyArray()
    ): UpdateDidContext {
        require(did is CanonicalPrismDid) {
            "DID should be in canonical form, found DID: $did"
        }

        val atalaUpdateOperation = AtalaOperation(
            operation = AtalaOperation.Operation.UpdateDid(
                updateDid = UpdateDIDOperation(
                    previousOperationHash = ByteArr(previousHash.value),
                    id = did.suffix,
                    actions = keysToAdd.map { keyInformation ->
                        UpdateDIDAction(
                            UpdateDIDAction.Action.AddKey(
                                AddKeyAction(keyInformation.toProto())
                            )
                        )
                    } + keysToRevoke.map { keyToRevoke ->
                        UpdateDIDAction(
                            UpdateDIDAction.Action.RemoveKey(
                                RemoveKeyAction(keyToRevoke)
                            )
                        )
                    }
                )
            )
        )

        return UpdateDidContext(
            operationHash = Sha256.compute(atalaUpdateOperation.encodeToByteArray()),
            updateDidSignedOperation = ProtoUtils.signedAtalaOperation(
                privateKey = signingPrivateKey,
                signedWith = signingKeyId,
                atalaOperation = atalaUpdateOperation
            )
        )
    }
}
