package io.iohk.atala.prism.api.models

import io.iohk.atala.prism.common.PrismSdkInternal
import io.iohk.atala.prism.protos.OperationStatus
import kotlin.js.JsExport

public typealias AtalaOperationStatusEnum = Int

// TODO: This smell-code is because Kotlin/JS doesn't support interfaces and enum classes now.
// We should switch to a enum class once Kotlin/JS fixes the issue: https://youtrack.jetbrains.com/issue/KT-37916.
/**
 * This class represents status of an operation submitted to PRISM Node.
 *
 * - UNKNOWN_OPERATION means that the operation is unknown to the Node
 * - PENDING_SUBMISSION means that the operation has not been submitted to Cardano ledger yet
 * - AWAIT_CONFIRMATION means that the operation has been submitted to Cardano ledger, but not
 * confirmed yet
 * - CONFIRMED_AND_APPLIED means that the operation has been confirmed by Cardano ledger and is
 * successfully applied by PRISM Node
 * - CONFIRMED_AND_REJECTED means that the operation has been confirmed by Cardano ledger, but was
 * rejected by PRISM Node
 */
@JsExport
public object AtalaOperationStatus {
    private val values: List<String> = listOf(
        "UNKNOWN_OPERATION",
        "PENDING_SUBMISSION",
        "AWAIT_CONFIRMATION",
        "CONFIRMED_AND_APPLIED",
        "CONFIRMED_AND_REJECTED"
    )

    public val UNKNOWN_OPERATION: Int = 0
    public val PENDING_SUBMISSION: Int = 1
    public val AWAIT_CONFIRMATION: Int = 2
    public val CONFIRMED_AND_APPLIED: Int = 3
    public val CONFIRMED_AND_REJECTED: Int = 4

    public fun asString(key: AtalaOperationStatusEnum): String {
        require(0 <= key && key < values.size)
        return values[key]
    }

    public fun fromString(string: String): AtalaOperationStatusEnum? {
        val index = values.indexOf(string)
        return if (index == -1)
            null
        else
            index
    }
}

@PrismSdkInternal
public fun OperationStatus.toModel(): AtalaOperationStatusEnum {
    return AtalaOperationStatus.fromString(name!!)!!
}

@PrismSdkInternal
public fun AtalaOperationStatusEnum.toProto(): OperationStatus {
    return OperationStatus.fromName(AtalaOperationStatus.asString(this))
}
