package io.iohk.atala.prism.api.connector

import io.iohk.atala.prism.api.PrismPayload
import io.iohk.atala.prism.api.common.DidValidation
import io.iohk.atala.prism.api.models.AtalaOperationId
import io.iohk.atala.prism.identity.LongFormPrismDid
import io.iohk.atala.prism.identity.PrismDid
import io.iohk.atala.prism.protos.*
import io.ktor.utils.io.core.*
import pbandk.ByteArr

/**
 * @suppress
 */
public class ConnectorPublicApiImpl internal constructor(
    private val client: ConnectorServiceCoroutine,
) : ConnectorPublicApi, Closeable {

    private fun toAtalaOperationId(byteArr: ByteArr): AtalaOperationId =
        AtalaOperationId.fromBytes(byteArr.array)

    /**
     * @param options gRPC options pointing to the running PRISM Connector service
     */
    public constructor(
        options: GrpcOptions
    ) : this(ConnectorServiceCoroutine.Client(GrpcClient(options)))

    override suspend fun registerExistingDid(did: PrismDid, name: String, role: Role): AtalaOperationId {
        return registerDid(RegisterDIDRequest.RegisterWith.ExistingDid(did.value), name, role)
    }

    override suspend fun healthCheck() {
        client.HealthCheck(HealthCheckRequest())
    }

    override suspend fun registerNewDid(
        payload: PrismPayload,
        did: LongFormPrismDid?,
        masterKeyId: String?,
        name: String,
        role: Role
    ): AtalaOperationId {
        val signedAtalaOperation = DidValidation.verifyPayload(did, masterKeyId, payload)
        return registerDid(RegisterDIDRequest.RegisterWith.CreateDidOperation(signedAtalaOperation), name, role)
    }

    private suspend fun registerDid(
        registerDIDRequest: RegisterDIDRequest.RegisterWith<*>,
        name: String,
        role: Role
    ) = toAtalaOperationId(
        client.RegisterDID(
            RegisterDIDRequest(
                registerWith = registerDIDRequest,
                role = when (role) {
                    is Role.ISSUER -> RegisterDIDRequest.Role.ISSUER
                    is Role.VERIFIER -> RegisterDIDRequest.Role.VERIFIER
                    is Role.UNRECOGNIZED -> RegisterDIDRequest.Role.UNRECOGNIZED(role.value)
                },
                name = name
            )
        ).operationId
    )

    override suspend fun getConnectionTokenInfo(token: String): ConnectorTokenInfo {
        val response = client.GetConnectionTokenInfo(GetConnectionTokenInfoRequest(token))
        return ConnectorTokenInfo(
            ConnectorParticipantInfo(
                name = response.creatorName,
                did = PrismDid.fromString(response.creatorDid),
                logo = response.creatorLogo.array
            )
        )
    }

    public override fun close() {
        client.close()
    }
}
