package io.iohk.atala.prism.api.connector

import io.iohk.atala.prism.api.common.AuthDetails
import io.iohk.atala.prism.api.common.authenticatedApiCall
import io.iohk.atala.prism.crypto.keys.ECPublicKey
import io.iohk.atala.prism.identity.PrismDid
import io.iohk.atala.prism.protos.*
import io.ktor.utils.io.core.*
import pbandk.ByteArr
import pbandk.encodeToByteArray

/**
 * @suppress
 */
public class ConnectorAuthApiImpl internal constructor(
    private val client: ConnectorServiceCoroutine,
) : ConnectorAuthApi, ConnectorPublicApi by ConnectorPublicApiImpl(client), Closeable {

    /**
     * @param options gRPC options pointing to the running PRISM Connector service
     */
    public constructor(
        options: GrpcOptions
    ) : this(ConnectorServiceCoroutine.Client(GrpcClient(options)))

    override suspend fun generateConnectionToken(
        count: Int,
        authDetails: AuthDetails
    ): List<String> =
        authenticatedApiCall(
            GenerateConnectionTokenRequest(count),
            authDetails
        ) { request, prismMetadata -> client.GenerateConnectionTokenAuth(request, prismMetadata) }.tokens

    override suspend fun addConnectionFromToken(
        token: String,
        publicKey: ECPublicKey,
        authDetails: AuthDetails
    ): ConnectorConnectionInfo =
        authenticatedApiCall(
            AddConnectionFromTokenRequest(token),
            authDetails
        ) { request, prismMetadata -> client.AddConnectionFromTokenAuth(request, prismMetadata) }
            .let {
                val connection = it.connection ?: throw IllegalStateException("Received empty connection.")
                ConnectorConnectionInfo(
                    ConnectorConnectionReference(
                        id = connection.connectionId,
                        token = connection.token,
                    ),
                    created = connection.created,
                    ConnectorParticipantInfo(
                        name = connection.participantName,
                        did = PrismDid.fromString(connection.participantDid),
                        connection.participantLogo.array
                    )
                )
            }

    override suspend fun getConnectionByToken(token: String, authDetails: AuthDetails): ConnectorConnectionReference? =
        authenticatedApiCall(
            GetConnectionByTokenRequest(token),
            authDetails
        ) { request, prismMetadata -> client.GetConnectionByTokenAuth(request, prismMetadata) }
            .connection?.let {
                ConnectorConnectionReference(
                    id = it.connectionId,
                    token = it.connectionToken,
                )
            }

    override suspend fun sendMessage(
        connectionId: String,
        message: AtalaMessage,
        id: String,
        authDetails: AuthDetails
    ): SentMessageReference {
        return authenticatedApiCall(
            SendMessageRequest(connectionId, ByteArr(message.encodeToByteArray()), id),
            authDetails
        ) { request, prismMetadata -> client.SendMessageAuth(request, prismMetadata) }
            .id.let {
                SentMessageReference(it)
            }
    }

    public override fun close() {
        client.close()
    }
}
