package io.iohk.atala.prism.api.common

import io.iohk.atala.prism.api.PrismPayload
import io.iohk.atala.prism.crypto.EC
import io.iohk.atala.prism.identity.LongFormPrismDid
import io.iohk.atala.prism.identity.findPublicKey
import io.iohk.atala.prism.identity.toModel
import io.iohk.atala.prism.protos.SignedAtalaOperation
import pbandk.decodeFromByteArray
import pbandk.encodeToByteArray
import kotlin.js.JsExport

@JsExport
public object DidValidation {

    public fun verifyPayload(
        did: LongFormPrismDid?,
        masterKeyId: String?,
        payload: PrismPayload
    ): SignedAtalaOperation {
        val signedAtalaOperation = SignedAtalaOperation.decodeFromByteArray(payload.value)
        val operationBytes = signedAtalaOperation.operation?.encodeToByteArray()
        require(operationBytes != null) {
            "Payload does not contain a valid Atala operation"
        }

        if (did != null && masterKeyId != null) {
            val masterKeyInfo =
                did.initialState.createDid?.didData?.toModel(did.asCanonical())?.publicKeys?.findPublicKey(masterKeyId)
            require(masterKeyInfo != null) {
                "Could not find key with id '$masterKeyId' in DID '$did'"
            }
            val signatureBytes = signedAtalaOperation.signature.array
            val signature = try {
                EC.toSignatureFromBytes(signatureBytes)
            } catch (e: Exception) {
                throw IllegalArgumentException("Payload does not contain a valid signature", e)
            }
            require(EC.verifyBytes(operationBytes, masterKeyInfo.didPublicKey.publicKey, signature)) {
                "Payload signature does not correspond to its content"
            }
        }

        if (did != null) {
            val createDidOperation = signedAtalaOperation.operation?.createDid
            require(createDidOperation != null) {
                "Payload operation is not of right type"
            }
            val prismDidDataModel = createDidOperation.didData?.toModel(did.asCanonical())
            require(prismDidDataModel != null) {
                "Payload operation does not contain valid information about DID data model"
            }
            require(prismDidDataModel.did == did.asCanonical()) {
                "Expected payload to contain an operation for creation of DID '${did.asCanonical()}', but got '${prismDidDataModel.did}'"
            }
        }

        return signedAtalaOperation
    }
}
