package io.iohk.atala.prism.api

import io.iohk.atala.prism.common.Validated
import io.iohk.atala.prism.credentials.PrismCredential
import io.iohk.atala.prism.credentials.json.JsonBasedCredential
import io.iohk.atala.prism.crypto.MerkleInclusionProof
import kotlinx.serialization.json.*
import kotlin.js.JsExport
import kotlin.js.JsName
import kotlin.jvm.JvmStatic

/**
 * Represents a [PrismCredential] along with its [MerkleInclusionProof].
 */
@JsExport
public data class VerifiableCredential(
    val signedCredential: PrismCredential,
    val merkleInclusionProof: MerkleInclusionProof
) {

    /**
     * Method that exports a verifiable credential as String
     * @param signedCredential
     * @param merkleInclusionProof
     */
    public fun export(): String =
        JsonObject(
            mapOf(
                Pair(Companion.encodedSignedCredentialField, JsonPrimitive(signedCredential.canonicalForm)),
                Pair(Companion.proofField, merkleInclusionProof.toJson())
            )
        ).toString()

    public companion object {
        private const val encodedSignedCredentialField = "encodedSignedCredential"
        private const val proofField = "proof"

        /**
         * Method that extracts a verifiable credential from a String
         * @param encodedVerifiableCredential example expected format: { "encodedSignedCredential": "eyJpZCI6ImRpZDpwcmlzbTplZTRmNmY3MzdhNThlOGViYTg3OWY5NjUzYmY1ZjMwZjcyOTc4MTI5NGI5ODk0OTQ4NDFhZTQ1NDhmYTIwMzdkIiwia2V5SWQiOiJpc3N1aW5nMCIsImNyZWRlbnRpYWxTdWJqZWN0Ijp7Im5hbWUiOiJDaGFybGllIiwiZGVncmVlIjoiQXRhbGEgUHJpc20gUGlvbmVlciIsInllYXIiOjIwMjEsImlkIjoiZGlkOnByaXNtOjQyY2E4NGVlMzNhODA4MWQ0OTk1ZTNkOTRjNzI2ZjAwZDIyYWJkYTc1OGVkOGY2OTJlNmYxOGU1Y2ZkZDJlMjA6Q2o4S1BSSTdDZ2R0WVhOMFpYSXdFQUZLTGdvSmMyVmpjREkxTm1zeEVpRUQ0N21FaXNVQzJVREFkZ3BOMUFEcEhhY2F6Wk5FVTBka3c0YUVXeGR0aGNzIn19.MEQCIHlpYh7jU47wc_mNXhRnRR_jeV9tmv9kg4kC8R8HBsWbAiBWM8TUtEFSebFe5bR9_k83HQQmtH0-XKRnuxARju970A", "proof": {"hash":"af09c3f0f93074a571adf9be95881eb5378c98f898dffea91e1fbc0c8ee74fdf","index":1,"siblings":["3c21a60e2540286e730b2cb16543eb0b85252a5df322c343627634398e8122ac"]} }}
         */
        @JvmStatic
        @JsName("importVerifiableCredential")
        public fun import(encodedVerifiableCredential: String): VerifiableCredential {
            val unparsableInputError =
                """Unable to parse provided encodedVerifiableCredential to Json. Please check format. encodedVerifiableCredential=$encodedVerifiableCredential"""
            val validatedJson =
                Validated.tryOrInvalid(
                    { Json.parseToJsonElement(encodedVerifiableCredential) },
                    unparsableInputError
                ).flatMap {
                    Validated.getOrError(
                        it as? JsonObject,
                        unparsableInputError
                    )
                }
            val validatedSignedCredential =
                validatedJson.flatMap { json ->
                    Validated.getOrError(
                        json[Companion.encodedSignedCredentialField]?.jsonPrimitive?.content,
                        """$encodedSignedCredentialField field is missing from provided encodedVerifiableCredential. encodedVerifiableCredential=$encodedVerifiableCredential"""
                    )
                }.map { JsonBasedCredential.fromString(it) }

            val validatedProof =
                validatedJson.flatMap { json ->
                    Validated.getOrError(
                        json[Companion.proofField]?.jsonObject,
                        """$proofField field is missing from provided encodedVerifiableCredential. encodedVerifiableCredential=$encodedVerifiableCredential"""
                    )
                }.flatMap { MerkleInclusionProof.decodeJson(it) }

            return Validated.Applicative.apply(
                validatedSignedCredential,
                validatedProof,
            ) { signedCredential, proof -> Validated.Valid(VerifiableCredential(signedCredential, proof)) }
                .getElseThrow { IllegalArgumentException(it) }
        }
    }
}
