package io.iohk.atala.prism.api

import io.iohk.atala.prism.api.models.ProtocolVersion
import io.iohk.atala.prism.common.PrismSdkInternal
import io.iohk.atala.prism.credentials.CredentialBatchId
import io.iohk.atala.prism.credentials.PrismCredential
import io.iohk.atala.prism.crypto.EC
import io.iohk.atala.prism.crypto.Sha256Digest
import io.iohk.atala.prism.crypto.keys.ECPrivateKey
import io.iohk.atala.prism.identity.CanonicalPrismDid
import io.iohk.atala.prism.protos.*
import io.iohk.atala.prism.protos.util.Base64Utils
import pbandk.ByteArr
import pbandk.decodeFromByteArray
import pbandk.encodeToByteArray
import kotlin.js.JsExport

@JsExport
@PrismSdkInternal
public object ProtoUtils {
    public fun signedAtalaOperation(
        privateKey: ECPrivateKey,
        signedWith: String,
        atalaOperation: AtalaOperation
    ): SignedAtalaOperation {
        val signature = EC.signBytes(atalaOperation.encodeToByteArray(), privateKey)
        return SignedAtalaOperation(
            signedWith = signedWith,
            signature = ByteArr(signature.getEncoded()),
            operation = atalaOperation
        )
    }

    public fun issueCredentialBatchOperation(data: CredentialBatchData): AtalaOperation {
        val isssueCredentialBatch = IssueCredentialBatchOperation(data)
        return AtalaOperation(AtalaOperation.Operation.IssueCredentialBatch(isssueCredentialBatch))
    }

    public fun revokeCredentialsOperation(
        batchOperationHash: Sha256Digest,
        batchId: CredentialBatchId,
        credentials: List<PrismCredential>
    ): AtalaOperation {
        val revokeCredentialsService = RevokeCredentialsOperation(
            previousOperationHash = ByteArr(batchOperationHash.value),
            credentialBatchId = batchId.id,
            credentialsToRevoke = credentials.map { ByteArr(it.hash().value) }
        )
        return AtalaOperation(AtalaOperation.Operation.RevokeCredentials(revokeCredentialsService))
    }

    public fun revokeCredentialsSignedOperation(
        revocationKKeyId: String,
        revocationKey: ECPrivateKey,
        issuanceOperationHash: Sha256Digest,
        batchId: String,
        credentialsToRevoke: List<Sha256Digest> = emptyList()
    ): SignedAtalaOperation {

        val revokeCredentialsService = RevokeCredentialsOperation(
            previousOperationHash = ByteArr(issuanceOperationHash.value),
            credentialBatchId = batchId,
            credentialsToRevoke = credentialsToRevoke.map { ByteArr(it.value) }
        )
        return signedAtalaOperation(
            revocationKey,
            revocationKKeyId,
            AtalaOperation(AtalaOperation.Operation.RevokeCredentials(revokeCredentialsService))
        )
    }

    public fun protocolUpdateSignedOperation(
        masterKeyId: String,
        masterKey: ECPrivateKey,
        did: CanonicalPrismDid,
        versionName: String?,
        protocolVersion: ProtocolVersion,
        effectiveSinceBlockIndex: Int
    ): SignedAtalaOperation {

        val protocolVersionUpdateOp = ProtocolVersionUpdateOperation(
            proposerDid = did.suffix,
            version = ProtocolVersionInfo(
                versionName = versionName ?: "",
                protocolVersion = io.iohk.atala.prism.protos.ProtocolVersion(
                    majorVersion = protocolVersion.majorVersion,
                    minorVersion = protocolVersion.minorVersion
                ),
                effectiveSince = effectiveSinceBlockIndex
            )
        )
        return signedAtalaOperation(
            masterKey,
            masterKeyId,
            AtalaOperation(AtalaOperation.Operation.ProtocolVersionUpdate(protocolVersionUpdateOp))
        )
    }

    public fun deactivateDIDOperation(
        masterKeyId: String,
        masterKey: ECPrivateKey,
        did: CanonicalPrismDid,
        previousOperationHash: Sha256Digest
    ): SignedAtalaOperation {

        val deactivateDIDOp = DeactivateDIDOperation(
            id = did.suffix,
            previousOperationHash = ByteArr(previousOperationHash.value)
        )
        return signedAtalaOperation(
            masterKey,
            masterKeyId,
            AtalaOperation(AtalaOperation.Operation.DeactivateDid(deactivateDIDOp))
        )
    }

    public fun signedAtalaOperationEncode(signedAtalaOperation: SignedAtalaOperation): String =
        Base64Utils.encode(signedAtalaOperation.encodeToByteArray())

    public fun signedAtalaOperationDecode(encodedOperation: String): SignedAtalaOperation =
        SignedAtalaOperation.decodeFromByteArray(Base64Utils.decode(encodedOperation))
}
